// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { immutableClass, KeyringTraceFlag, needs, readOnlyProperty, } from '@aws-crypto/material-management';
import { constructArnInOtherRegion, isMultiRegionAwsKmsArn, parseAwsKmsKeyArn, } from './arn_parsing';
import { decrypt, KMS_PROVIDER_ID } from './helpers';
export function AwsKmsMrkAwareSymmetricDiscoveryKeyringClass(BaseKeyring) {
    class AwsKmsMrkAwareSymmetricDiscoveryKeyring
    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.5
    //# MUST implement that AWS Encryption SDK Keyring interface (../keyring-
    //# interface.md#interface)
     extends BaseKeyring {
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.6
        //# On initialization the caller MUST provide:
        constructor({ client, grantTokens, discoveryFilter, }) {
            super();
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.6
            //# The keyring MUST know what Region the AWS KMS client is in.
            //
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.6
            //# It
            //# SHOULD obtain this information directly from the client as opposed to
            //# having an additional parameter.
            //
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.6
            //# However if it can not, then it MUST
            //# NOT create the client itself.
            //
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.6
            //# It SHOULD have a Region parameter and
            //# SHOULD try to identify mismatched configurations.
            //
            // @ts-ignore the V3 client has set the config to protected
            const clientRegion = client.config.region;
            needs(clientRegion, 'Client must be configured to a region.');
            /* Precondition: The AwsKmsMrkAwareSymmetricDiscoveryKeyring Discovery filter *must* be able to match something.
             * I am not going to wait to tell you
             * that no CMK can match an empty account list.
             * e.g. [], [''], '' are not valid.
             */
            needs(!discoveryFilter ||
                (discoveryFilter.accountIDs &&
                    discoveryFilter.accountIDs.length &&
                    !!discoveryFilter.partition &&
                    discoveryFilter.accountIDs.every((a) => typeof a === 'string' && !!a)), 'A discovery filter must be able to match something.');
            readOnlyProperty(this, 'client', client);
            // AWS SDK v3 stores the clientRegion behind an async function
            if (typeof clientRegion == 'function') {
                /* Postcondition: Store the AWS SDK V3 region promise as the clientRegion.
                 * This information MUST be communicated to OnDecrypt.
                 * If a caller creates a keyring,
                 * and then calls OnDecrypt all in synchronous code
                 * then the v3 region will not have been able to resolve.
                 * If clientRegion was null,
                 * then the keyring would filter out all EDKs
                 * because the region does not match.
                 */
                this.clientRegion = clientRegion().then((region) => {
                    /* Postcondition: Resolve the AWS SDK V3 region promise and update clientRegion. */
                    readOnlyProperty(this, 'clientRegion', region);
                    /* Postcondition: Resolve the promise with the value set. */
                    return region;
                });
            }
            else {
                readOnlyProperty(this, 'clientRegion', clientRegion);
            }
            readOnlyProperty(this, 'grantTokens', grantTokens);
            readOnlyProperty(this, 'discoveryFilter', discoveryFilter
                ? Object.freeze({
                    ...discoveryFilter,
                    accountIDs: Object.freeze(discoveryFilter.accountIDs),
                })
                : discoveryFilter);
        }
        async _onEncrypt() {
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.7
            //# This function MUST fail.
            throw new Error('AwsKmsMrkAwareSymmetricDiscoveryKeyring cannot be used to encrypt');
        }
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# OnDecrypt MUST take decryption materials (structures.md#decryption-
        //# materials) and a list of encrypted data keys
        //# (structures.md#encrypted-data-key) as input.
        async _onDecrypt(material, encryptedDataKeys) {
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
            //# If the decryption materials (structures.md#decryption-materials)
            //# already contained a valid plaintext data key OnDecrypt MUST
            //# immediately return the unmodified decryption materials
            //# (structures.md#decryption-materials).
            if (material.hasValidKey())
                return material;
            // See the constructor, this is to support both AWS SDK v2 and v3.
            needs(typeof this.clientRegion === 'string' ||
                /* Precondition: AWS SDK V3 region promise MUST have resolved to a string.
                 * In the constructor the region promise resolves
                 * to the same value that is then set.
                 */
                // @ts-ignore
                typeof (await this.clientRegion) == 'string', 'clientRegion MUST be a string.');
            const { client, grantTokens, clientRegion } = this;
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
            //# The set of encrypted data keys MUST first be filtered to match this
            //# keyring's configuration.
            const decryptableEDKs = encryptedDataKeys.filter(filterEDKs(this));
            const cmkErrors = [];
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
            //# For each encrypted data key in the filtered set, one at a time, the
            //# OnDecrypt MUST attempt to decrypt the data key.
            for (const edk of decryptableEDKs) {
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                //# Otherwise it MUST
                //# be the provider info.
                let keyId = edk.providerInfo;
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                //# *  "KeyId": If the provider info's resource type is "key" and its
                //# resource is a multi-Region key then a new ARN MUST be created
                //# where the region part MUST equal the AWS KMS client region and
                //# every other part MUST equal the provider info.
                const keyArn = parseAwsKmsKeyArn(edk.providerInfo);
                needs(keyArn, 'Unexpected EDK ProviderInfo for AWS KMS EDK');
                if (isMultiRegionAwsKmsArn(keyArn)) {
                    keyId = constructArnInOtherRegion(keyArn, clientRegion);
                }
                let dataKey = false;
                try {
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# When calling AWS KMS Decrypt
                    //# (https://docs.aws.amazon.com/kms/latest/APIReference/
                    //# API_Decrypt.html), the keyring MUST call with a request constructed
                    //# as follows:
                    dataKey = await decrypt(
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# To attempt to decrypt a particular encrypted data key
                    //# (structures.md#encrypted-data-key), OnDecrypt MUST call AWS KMS
                    //# Decrypt (https://docs.aws.amazon.com/kms/latest/APIReference/
                    //# API_Decrypt.html) with the configured AWS KMS client.
                    client, {
                        providerId: edk.providerId,
                        providerInfo: keyId,
                        encryptedDataKey: edk.encryptedDataKey,
                    }, material.encryptionContext, grantTokens);
                    /* This should be impossible given that decrypt only returns false if the client supplier does
                     * or if the providerId is not "aws-kms", which we have already filtered out
                     */
                    if (!dataKey)
                        continue;
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# *  The "KeyId" field in the response MUST equal the requested "KeyId"
                    needs(dataKey.KeyId === keyId, 'KMS Decryption key does not match the requested key id.');
                    const flags = KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY |
                        KeyringTraceFlag.WRAPPING_KEY_VERIFIED_ENC_CTX;
                    const trace = {
                        keyNamespace: KMS_PROVIDER_ID,
                        keyName: dataKey.KeyId,
                        flags,
                    };
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# *  The length of the response's "Plaintext" MUST equal the key
                    //# derivation input length (algorithm-suites.md#key-derivation-input-
                    //# length) specified by the algorithm suite (algorithm-suites.md)
                    //# included in the input decryption materials
                    //# (structures.md#decryption-materials).
                    //
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# Since the response does satisfies these requirements then OnDecrypt
                    //# MUST do the following with the response:
                    //
                    // setUnencryptedDataKey will throw if the plaintext does not match the algorithm suite requirements.
                    material.setUnencryptedDataKey(dataKey.Plaintext, trace);
                    return material;
                }
                catch (e) {
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# If the response does not satisfies these requirements then an error
                    //# is collected and the next encrypted data key in the filtered set MUST
                    //# be attempted.
                    cmkErrors.push({ errPlus: e });
                }
            }
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
            //# If OnDecrypt fails to successfully decrypt any encrypted data key
            //# (structures.md#encrypted-data-key), then it MUST yield an error that
            //# includes all collected errors.
            needs(material.hasValidKey(), [
                `Unable to decrypt data key${!decryptableEDKs.length ? ': No EDKs supplied' : ''}.`,
                ...cmkErrors.map((e, i) => `Error #${i + 1}  \n${e.errPlus.stack}`),
            ].join('\n'));
            return material;
        }
    }
    immutableClass(AwsKmsMrkAwareSymmetricDiscoveryKeyring);
    return AwsKmsMrkAwareSymmetricDiscoveryKeyring;
}
function filterEDKs({ discoveryFilter, clientRegion, }) {
    return function filter({ providerId, providerInfo }) {
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# *  Its provider ID MUST exactly match the value "aws-kms".
        if (providerId !== KMS_PROVIDER_ID)
            return false;
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# *  The provider info MUST be a valid AWS KMS ARN (aws-kms-key-
        //# arn.md#a-valid-aws-kms-arn) with a resource type of "key" or
        //# OnDecrypt MUST fail.
        const edkArn = parseAwsKmsKeyArn(providerInfo);
        needs(edkArn && edkArn.ResourceType === 'key', 'Unexpected EDK ProviderInfo for AWS KMS EDK');
        const { AccountId: account, Partition: partition, Region: edkRegion, } = edkArn;
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# *  If the provider info is not identified as a multi-Region key (aws-
        //# kms-key-arn.md#identifying-an-aws-kms-multi-region-key), then the
        //# provider info's Region MUST match the AWS KMS client region.
        if (!isMultiRegionAwsKmsArn(edkArn) && clientRegion !== edkRegion) {
            return false;
        }
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# *  If a discovery filter is configured, its partition and the
        //# provider info partition MUST match.
        //
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# *  If a discovery filter is configured, its set of accounts MUST
        //# contain the provider info account.
        return (!discoveryFilter ||
            (discoveryFilter.partition === partition &&
                discoveryFilter.accountIDs.includes(account)));
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia21zX21ya19kaXNjb3Zlcnlfa2V5cmluZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9rbXNfbXJrX2Rpc2NvdmVyeV9rZXlyaW5nLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBLG9FQUFvRTtBQUNwRSxzQ0FBc0M7QUFFdEMsT0FBTyxFQUlMLGNBQWMsRUFHZCxnQkFBZ0IsRUFDaEIsS0FBSyxFQUNMLGdCQUFnQixHQUlqQixNQUFNLGlDQUFpQyxDQUFBO0FBQ3hDLE9BQU8sRUFDTCx5QkFBeUIsRUFDekIsc0JBQXNCLEVBQ3RCLGlCQUFpQixHQUNsQixNQUFNLGVBQWUsQ0FBQTtBQUN0QixPQUFPLEVBQUUsT0FBTyxFQUFFLGVBQWUsRUFBRSxNQUFNLFdBQVcsQ0FBQTtBQXlDcEQsTUFBTSxVQUFVLDRDQUE0QyxDQUkxRCxXQUFnQztJQUVoQyxNQUFNLHVDQUF1QztJQUMzQyw2RkFBNkY7SUFDN0YseUVBQXlFO0lBQ3pFLDJCQUEyQjtJQUMzQixTQUFRLFdBQVc7UUFXbkIsNkZBQTZGO1FBQzdGLDhDQUE4QztRQUM5QyxZQUFZLEVBQ1YsTUFBTSxFQUNOLFdBQVcsRUFDWCxlQUFlLEdBQ3NDO1lBQ3JELEtBQUssRUFBRSxDQUFBO1lBRVAsNkZBQTZGO1lBQzdGLCtEQUErRDtZQUMvRCxFQUFFO1lBQ0YsNkZBQTZGO1lBQzdGLE1BQU07WUFDTix5RUFBeUU7WUFDekUsbUNBQW1DO1lBQ25DLEVBQUU7WUFDRiw2RkFBNkY7WUFDN0YsdUNBQXVDO1lBQ3ZDLGlDQUFpQztZQUNqQyxFQUFFO1lBQ0YsNkZBQTZGO1lBQzdGLHlDQUF5QztZQUN6QyxxREFBcUQ7WUFDckQsRUFBRTtZQUNGLDJEQUEyRDtZQUMzRCxNQUFNLFlBQVksR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQTtZQUN6QyxLQUFLLENBQUMsWUFBWSxFQUFFLHdDQUF3QyxDQUFDLENBQUE7WUFFN0Q7Ozs7ZUFJRztZQUNILEtBQUssQ0FDSCxDQUFDLGVBQWU7Z0JBQ2QsQ0FBQyxlQUFlLENBQUMsVUFBVTtvQkFDekIsZUFBZSxDQUFDLFVBQVUsQ0FBQyxNQUFNO29CQUNqQyxDQUFDLENBQUMsZUFBZSxDQUFDLFNBQVM7b0JBQzNCLGVBQWUsQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUM5QixDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsT0FBTyxDQUFDLEtBQUssUUFBUSxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQ3BDLENBQUMsRUFDTixxREFBcUQsQ0FDdEQsQ0FBQTtZQUVELGdCQUFnQixDQUFDLElBQUksRUFBRSxRQUFRLEVBQUUsTUFBTSxDQUFDLENBQUE7WUFFeEMsOERBQThEO1lBQzlELElBQUksT0FBTyxZQUFZLElBQUksVUFBVSxFQUFFO2dCQUNyQzs7Ozs7Ozs7bUJBUUc7Z0JBQ0gsSUFBSSxDQUFDLFlBQVksR0FBRyxZQUFZLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFjLEVBQUUsRUFBRTtvQkFDekQsbUZBQW1GO29CQUNuRixnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFLE1BQU0sQ0FBQyxDQUFBO29CQUM5Qyw0REFBNEQ7b0JBQzVELE9BQU8sTUFBTSxDQUFBO2dCQUNmLENBQUMsQ0FBQyxDQUFBO2FBQ0g7aUJBQU07Z0JBQ0wsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRSxZQUFZLENBQUMsQ0FBQTthQUNyRDtZQUVELGdCQUFnQixDQUFDLElBQUksRUFBRSxhQUFhLEVBQUUsV0FBVyxDQUFDLENBQUE7WUFDbEQsZ0JBQWdCLENBQ2QsSUFBSSxFQUNKLGlCQUFpQixFQUNqQixlQUFlO2dCQUNiLENBQUMsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDO29CQUNaLEdBQUcsZUFBZTtvQkFDbEIsVUFBVSxFQUFFLE1BQU0sQ0FBQyxNQUFNLENBQUMsZUFBZSxDQUFDLFVBQVUsQ0FBQztpQkFDdEQsQ0FBQztnQkFDSixDQUFDLENBQUMsZUFBZSxDQUNwQixDQUFBO1FBQ0gsQ0FBQztRQUVELEtBQUssQ0FBQyxVQUFVO1lBQ2QsNkZBQTZGO1lBQzdGLDRCQUE0QjtZQUM1QixNQUFNLElBQUksS0FBSyxDQUNiLG1FQUFtRSxDQUNwRSxDQUFBO1FBQ0gsQ0FBQztRQUVELDZGQUE2RjtRQUM3Rix1RUFBdUU7UUFDdkUsZ0RBQWdEO1FBQ2hELGdEQUFnRDtRQUNoRCxLQUFLLENBQUMsVUFBVSxDQUNkLFFBQStCLEVBQy9CLGlCQUFxQztZQUVyQyw2RkFBNkY7WUFDN0Ysb0VBQW9FO1lBQ3BFLCtEQUErRDtZQUMvRCwwREFBMEQ7WUFDMUQseUNBQXlDO1lBQ3pDLElBQUksUUFBUSxDQUFDLFdBQVcsRUFBRTtnQkFBRSxPQUFPLFFBQVEsQ0FBQTtZQUUzQyxrRUFBa0U7WUFDbEUsS0FBSyxDQUNILE9BQU8sSUFBSSxDQUFDLFlBQVksS0FBSyxRQUFRO2dCQUNuQzs7O21CQUdHO2dCQUNILGFBQWE7Z0JBQ2IsT0FBTyxDQUFDLE1BQU0sSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLFFBQVEsRUFDOUMsZ0NBQWdDLENBQ2pDLENBQUE7WUFFRCxNQUFNLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRSxZQUFZLEVBQUUsR0FBRyxJQUFJLENBQUE7WUFFbEQsNkZBQTZGO1lBQzdGLHVFQUF1RTtZQUN2RSw0QkFBNEI7WUFDNUIsTUFBTSxlQUFlLEdBQUcsaUJBQWlCLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFBO1lBQ2xFLE1BQU0sU0FBUyxHQUFnQixFQUFFLENBQUE7WUFFakMsNkZBQTZGO1lBQzdGLHVFQUF1RTtZQUN2RSxtREFBbUQ7WUFDbkQsS0FBSyxNQUFNLEdBQUcsSUFBSSxlQUFlLEVBQUU7Z0JBQ2pDLDZGQUE2RjtnQkFDN0YscUJBQXFCO2dCQUNyQix5QkFBeUI7Z0JBQ3pCLElBQUksS0FBSyxHQUFHLEdBQUcsQ0FBQyxZQUFZLENBQUE7Z0JBQzVCLDZGQUE2RjtnQkFDN0YscUVBQXFFO2dCQUNyRSxpRUFBaUU7Z0JBQ2pFLGtFQUFrRTtnQkFDbEUsa0RBQWtEO2dCQUNsRCxNQUFNLE1BQU0sR0FBRyxpQkFBaUIsQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLENBQUE7Z0JBQ2xELEtBQUssQ0FBQyxNQUFNLEVBQUUsNkNBQTZDLENBQUMsQ0FBQTtnQkFDNUQsSUFBSSxzQkFBc0IsQ0FBQyxNQUFNLENBQUMsRUFBRTtvQkFDbEMsS0FBSyxHQUFHLHlCQUF5QixDQUFDLE1BQU0sRUFBRSxZQUFZLENBQUMsQ0FBQTtpQkFDeEQ7Z0JBRUQsSUFBSSxPQUFPLEdBQW9DLEtBQUssQ0FBQTtnQkFDcEQsSUFBSTtvQkFDRiw2RkFBNkY7b0JBQzdGLGdDQUFnQztvQkFDaEMseURBQXlEO29CQUN6RCx1RUFBdUU7b0JBQ3ZFLGVBQWU7b0JBQ2YsT0FBTyxHQUFHLE1BQU0sT0FBTztvQkFDckIsNkZBQTZGO29CQUM3Rix5REFBeUQ7b0JBQ3pELG1FQUFtRTtvQkFDbkUsaUVBQWlFO29CQUNqRSx5REFBeUQ7b0JBQ3pELE1BQU0sRUFDTjt3QkFDRSxVQUFVLEVBQUUsR0FBRyxDQUFDLFVBQVU7d0JBQzFCLFlBQVksRUFBRSxLQUFLO3dCQUNuQixnQkFBZ0IsRUFBRSxHQUFHLENBQUMsZ0JBQWdCO3FCQUN2QyxFQUNELFFBQVEsQ0FBQyxpQkFBaUIsRUFDMUIsV0FBVyxDQUNaLENBQUE7b0JBQ0Q7O3VCQUVHO29CQUNILElBQUksQ0FBQyxPQUFPO3dCQUFFLFNBQVE7b0JBRXRCLDZGQUE2RjtvQkFDN0YseUVBQXlFO29CQUN6RSxLQUFLLENBQ0gsT0FBTyxDQUFDLEtBQUssS0FBSyxLQUFLLEVBQ3ZCLHlEQUF5RCxDQUMxRCxDQUFBO29CQUVELE1BQU0sS0FBSyxHQUNULGdCQUFnQixDQUFDLCtCQUErQjt3QkFDaEQsZ0JBQWdCLENBQUMsNkJBQTZCLENBQUE7b0JBQ2hELE1BQU0sS0FBSyxHQUFpQjt3QkFDMUIsWUFBWSxFQUFFLGVBQWU7d0JBQzdCLE9BQU8sRUFBRSxPQUFPLENBQUMsS0FBSzt3QkFDdEIsS0FBSztxQkFDTixDQUFBO29CQUVELDZGQUE2RjtvQkFDN0Ysa0VBQWtFO29CQUNsRSxzRUFBc0U7b0JBQ3RFLGtFQUFrRTtvQkFDbEUsOENBQThDO29CQUM5Qyx5Q0FBeUM7b0JBQ3pDLEVBQUU7b0JBQ0YsNkZBQTZGO29CQUM3Rix1RUFBdUU7b0JBQ3ZFLDRDQUE0QztvQkFDNUMsRUFBRTtvQkFDRixxR0FBcUc7b0JBQ3JHLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFBO29CQUN4RCxPQUFPLFFBQVEsQ0FBQTtpQkFDaEI7Z0JBQUMsT0FBTyxDQUFDLEVBQUU7b0JBQ1YsNkZBQTZGO29CQUM3Rix1RUFBdUU7b0JBQ3ZFLHlFQUF5RTtvQkFDekUsaUJBQWlCO29CQUNqQixTQUFTLENBQUMsSUFBSSxDQUFDLEVBQUUsT0FBTyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUE7aUJBQy9CO2FBQ0Y7WUFDRCw2RkFBNkY7WUFDN0YscUVBQXFFO1lBQ3JFLHdFQUF3RTtZQUN4RSxrQ0FBa0M7WUFDbEMsS0FBSyxDQUNILFFBQVEsQ0FBQyxXQUFXLEVBQUUsRUFDdEI7Z0JBQ0UsNkJBQ0UsQ0FBQyxlQUFlLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLENBQUMsRUFDbkQsR0FBRztnQkFDSCxHQUFHLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsQ0FBQzthQUNwRSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FDYixDQUFBO1lBQ0QsT0FBTyxRQUFRLENBQUE7UUFDakIsQ0FBQztLQUNGO0lBQ0QsY0FBYyxDQUFDLHVDQUF1QyxDQUFDLENBQUE7SUFDdkQsT0FBTyx1Q0FBdUMsQ0FBQTtBQUNoRCxDQUFDO0FBRUQsU0FBUyxVQUFVLENBR2pCLEVBQ0EsZUFBZSxFQUNmLFlBQVksR0FDd0M7SUFDcEQsT0FBTyxTQUFTLE1BQU0sQ0FBQyxFQUFFLFVBQVUsRUFBRSxZQUFZLEVBQW9CO1FBQ25FLDZGQUE2RjtRQUM3Riw4REFBOEQ7UUFDOUQsSUFBSSxVQUFVLEtBQUssZUFBZTtZQUFFLE9BQU8sS0FBSyxDQUFBO1FBRWhELDZGQUE2RjtRQUM3RixrRUFBa0U7UUFDbEUsZ0VBQWdFO1FBQ2hFLHdCQUF3QjtRQUN4QixNQUFNLE1BQU0sR0FBRyxpQkFBaUIsQ0FBQyxZQUFZLENBQUMsQ0FBQTtRQUM5QyxLQUFLLENBQ0gsTUFBTSxJQUFJLE1BQU0sQ0FBQyxZQUFZLEtBQUssS0FBSyxFQUN2Qyw2Q0FBNkMsQ0FDOUMsQ0FBQTtRQUNELE1BQU0sRUFDSixTQUFTLEVBQUUsT0FBTyxFQUNsQixTQUFTLEVBQUUsU0FBUyxFQUNwQixNQUFNLEVBQUUsU0FBUyxHQUNsQixHQUFHLE1BQU0sQ0FBQTtRQUVWLDZGQUE2RjtRQUM3Rix5RUFBeUU7UUFDekUscUVBQXFFO1FBQ3JFLGdFQUFnRTtRQUNoRSxJQUFJLENBQUMsc0JBQXNCLENBQUMsTUFBTSxDQUFDLElBQUksWUFBWSxLQUFLLFNBQVMsRUFBRTtZQUNqRSxPQUFPLEtBQUssQ0FBQTtTQUNiO1FBRUQsNkZBQTZGO1FBQzdGLGlFQUFpRTtRQUNqRSx1Q0FBdUM7UUFDdkMsRUFBRTtRQUNGLDZGQUE2RjtRQUM3RixvRUFBb0U7UUFDcEUsc0NBQXNDO1FBQ3RDLE9BQU8sQ0FDTCxDQUFDLGVBQWU7WUFDaEIsQ0FBQyxlQUFlLENBQUMsU0FBUyxLQUFLLFNBQVM7Z0JBQ3RDLGVBQWUsQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQ2hELENBQUE7SUFDSCxDQUFDLENBQUE7QUFDSCxDQUFDIn0=