//
// Copyright 2019 Pixar
//
// Licensed under the terms set forth in the LICENSE.txt file available at
// https://openusd.org/license.
//

#include "pxr/pxr.h"
#include "pxr/base/tf/span.h"

#include "pxr/base/tf/diagnostic.h"
#include "pxr/base/tf/stringUtils.h"

#include <algorithm>


using namespace std;
PXR_NAMESPACE_USING_DIRECTIVE


template <typename Span, typename Container>
void
Tf_TestSpanMatchesContainer(const Span& span, const Container& cont)
{
    TF_AXIOM(span.data() == cont.data());
    TF_AXIOM(span.size() == cont.size());
    TF_AXIOM(std::equal(span.begin(), span.end(), cont.begin()));
    TF_AXIOM(std::equal(span.cbegin(), span.cend(), cont.cbegin()));
    TF_AXIOM(std::equal(span.rbegin(), span.rend(), cont.rbegin()));
    TF_AXIOM(std::equal(span.crbegin(), span.crend(), cont.crbegin()));
}


void Tf_TestImplicitConversionInOverloads(TfSpan<int> span) {}
void Tf_TestImplicitConversionInOverloads(TfSpan<float> span) {}

void Tf_TestConstImplicitConversionInOverloads(TfSpan<const int> span) {}
void Tf_TestConstImplicitConversionInOverloads(TfSpan<const float> span) {}
                  

int main(int argc, char** argv)
{
    // Test empty spans.
    {
        TfSpan<int> span;
        TF_AXIOM(span.empty());
        TF_AXIOM(span.size() == 0);
        TF_AXIOM(span.data() == nullptr);
    }

    const std::vector<int> constData({1,2,3,4,5});

    std::vector<int> data(constData);

    // Test construction of a const span from a non-const container.
    {
        TfSpan<const int> span = data;
        Tf_TestSpanMatchesContainer(span, data);

        // Copy constructor.
        TfSpan<const int> copy(span);
        Tf_TestSpanMatchesContainer(copy, span);
    }
    {
        auto span = TfMakeConstSpan(data);
        Tf_TestSpanMatchesContainer(span, data);
    }

    // Test construction of non-const span from a non-const container.
    {
        TfSpan<int> span = data;
        Tf_TestSpanMatchesContainer(span, data);
        TF_AXIOM(!span.empty());

        // Should be allowed to implicitly convert a non-const span to const.
        TfSpan<const int> cspan(span);
        Tf_TestSpanMatchesContainer(cspan, span);
    }
    {
        auto span = TfMakeSpan(data);
        Tf_TestSpanMatchesContainer(span, data);
    }

    // Test construction of const span from a const container.
    {
        // XXX: Should not be allowed to construct a non-const span
        // from a const container.
        // TfSpan<int> span = data; // discards cv-qualifier

        TfSpan<const int> span = constData;
        Tf_TestSpanMatchesContainer(span, constData);
    }
    {
        // XXX: Should not be allowed to construct a non-const span
        // from a const container.
        // auto span = TfMakeSpan(data); // discards cv-qualifier

        auto span = TfMakeConstSpan(constData);
        Tf_TestSpanMatchesContainer(span, constData);
    }

    // Test element accessors.
    {
        const TfSpan<const int> span(data);

        for (size_t i = 0; i < data.size(); ++i) {
            TF_AXIOM(span[i] == data[i]);
        }

        TF_AXIOM(span.front() == data.front());
        TF_AXIOM(span.back() == data.back());
    }

    // Test subspans.
    {
        // Should be able to construct subspans from a constant span.
        const TfSpan<const int> span(data);

        // Test with default count (std::dynamic_extent)
        TfSpan<const int> subspan = span.subspan(2);
        const std::vector<int> expectedSubspan({3,4,5});
        TF_AXIOM(std::equal(subspan.begin(), subspan.end(),
                            expectedSubspan.begin()));

        // Test with explicit count
        TfSpan<const int> subspan2 = span.subspan(2, 2);
        const std::vector<int> expectedSubspan2({3,4});
        TF_AXIOM(std::equal(subspan2.begin(), subspan2.end(),   
                            expectedSubspan2.begin()));

        // Test first.
        TfSpan<const int> subspan3 = span.first(2);
        const std::vector<int> expectedSubspan3({1,2});
        TF_AXIOM(std::equal(subspan3.begin(), subspan3.end(),
                            expectedSubspan3.begin()));

        // Test last.
        TfSpan<const int> subspan4 = span.last(2);
        const std::vector<int> expectedSubspan4({4,5});
        TF_AXIOM(std::equal(subspan4.begin(), subspan4.end(),
                            expectedSubspan4.begin()));
    }

    // Test span edits.
    {
        TfSpan<int> span(data);
        for (size_t i = 0; i < span.size(); ++i) {
            span[i] = (i+1)*10;
        }
        
        const std::vector<int> expected({10,20,30,40,50});
        TF_AXIOM(std::equal(data.begin(), data.end(),
                            expected.begin()));
    }

    // Test implicit conversion in function calls with multiple overloads.
    Tf_TestImplicitConversionInOverloads(data);
    Tf_TestConstImplicitConversionInOverloads(constData);

    return 0;
}
