//
// Copyright 2025 Pixar
//
// Licensed under the terms set forth in the LICENSE.txt file available at
// https://openusd.org/license.
//
#include "pxr/pxr.h"
#include "pxr/base/tf/envSetting.h"
#include "pxr/base/tf/staticTokens.h"
#include "pxr/base/tf/stringUtils.h"
#include "pxr/usd/sdf/fileFormat.h"
#include "pxr/usd/sdf/layer.h"
#include "pxr/usd/sdf/usdaFileFormat.h"
#include "pxr/usd/pcp/dynamicFileFormatContext.h"
#include "pxr/usd/pcp/dynamicFileFormatInterface.h"

#include <string>
#include <unordered_set>

PXR_NAMESPACE_OPEN_SCOPE

using std::string;

#define TEST_PCP_ASSET_PATH_DYNAMIC_FILE_FORMAT_TOKENS       \
    ((Id, "Test_PcpAssetPathDynamicFileFormat")) \
    ((Version, "1.0")) \
    ((Target, "usd")) \
    ((Extension, "testassetpathpcpdynamic")) \
    ((AssetPath, "TestPcp_assetPath")) \
    ((ResolvedAssetPath, "TestPcp_resolvedAssetPath")) \
    ((AssetPathArray, "TestPcp_assetPathArray")) \
    ((ResolvedAssetPathArray, "TestPcp_resolvedAssetPathArray"))

TF_DECLARE_PUBLIC_TOKENS(Test_PcpAssetPathDynamicFileFormatPlugin_FileFormatTokens, 
                         TEST_PCP_ASSET_PATH_DYNAMIC_FILE_FORMAT_TOKENS);

TF_DEFINE_PUBLIC_TOKENS(
    Test_PcpAssetPathDynamicFileFormatPlugin_FileFormatTokens, 
    TEST_PCP_ASSET_PATH_DYNAMIC_FILE_FORMAT_TOKENS);

TF_DEFINE_ENV_SETTING(TEST_PCP_ASSET_PATH_DYNAMIC_FILE_FORMAT_TOKENS_USE_ATTRIBUTE_INPUTS, 
    false,
    "Set to true or 1 to have this dynamic file format inputs come from "
    "attribute default values instead of prim metadata fields. This allows "
    "to test that we can get equivalent functionality from both types of "
    "inputs for dynamic payloads.");

/// \class Test_PcpAssetPathDynamicFileFormatPlugin_FileFormat
/// 
/// A test dynamic file format that takes the inputs TestPcp_assetPath and 
/// TestPcp_assetPathArray and converts them into payload file format arguments
/// that include both the evaluated asset path (through variable expressions)
/// and the resolved asset path. The contents of layers generated by this file
/// format are irrelevant as this is just used by the test to test generating
/// resolved asset paths for file format arguments in dynamic payloads.
class Test_PcpAssetPathDynamicFileFormatPlugin_FileFormat : public SdfFileFormat,
    public PcpDynamicFileFormatInterface
{
public:
    // SdfFileFormat required overrides.
    bool CanRead(const std::string &file) const override;
    bool Read(SdfLayer *layer,
              const std::string& resolvedPath,
              bool metadataOnly) const override;

    // A required override for generating dynamic arguments that are 
    // particular to this file format.
    void ComposeFieldsForFileFormatArguments(
        const std::string& assetPath, 
        const PcpDynamicFileFormatContext& context,
        FileFormatArguments* args,
        VtValue *dependencyContextData) const override;

protected:
    SDF_FILE_FORMAT_FACTORY_ACCESS;

    virtual ~Test_PcpAssetPathDynamicFileFormatPlugin_FileFormat();
    Test_PcpAssetPathDynamicFileFormatPlugin_FileFormat();

    // Override that allow us to use anonymous layer identifiers for payloads
    // to thid file format and still have Read be called to process dynamic 
    // file format arguments that would generate its contents.
    bool _ShouldSkipAnonymousReload() const override {
        return false;
    }
    bool _ShouldReadAnonymousLayers() const override {
        return true;
    }

private:
    // Helper cache for testing the functionality that the contextId passed in
    // to ComposeFieldsForFileFormatArguments will match the one passed 
    // into CanFieldChangeAffectFileFormatArguments during change processing 
    // if the change should indeed cause a primIndex to recompose.
    mutable std::unordered_set<uint64_t> _contextIds;
};

TF_REGISTRY_FUNCTION(TfType)
{
    SDF_DEFINE_FILE_FORMAT(Test_PcpAssetPathDynamicFileFormatPlugin_FileFormat, 
                           SdfFileFormat);
}

Test_PcpAssetPathDynamicFileFormatPlugin_FileFormat
    ::Test_PcpAssetPathDynamicFileFormatPlugin_FileFormat()
    : SdfFileFormat(
        Test_PcpAssetPathDynamicFileFormatPlugin_FileFormatTokens->Id,
        Test_PcpAssetPathDynamicFileFormatPlugin_FileFormatTokens->Version,
        Test_PcpAssetPathDynamicFileFormatPlugin_FileFormatTokens->Target,
        Test_PcpAssetPathDynamicFileFormatPlugin_FileFormatTokens->Extension)
{
}

Test_PcpAssetPathDynamicFileFormatPlugin_FileFormat
    ::~Test_PcpAssetPathDynamicFileFormatPlugin_FileFormat()
{
}

bool
Test_PcpAssetPathDynamicFileFormatPlugin_FileFormat::CanRead(
    const string& filePath) const
{
    return true;
}

bool
Test_PcpAssetPathDynamicFileFormatPlugin_FileFormat::Read(
    SdfLayer *layer,
    const string& resolvedPathIn,
    bool metadataOnly) const
{
    if (!TF_VERIFY(layer)) {
        return false;
    }

    // The contents of the layer are irrelevant for the test that uses this
    // file format which is only checking the expected file format arguments for
    // a dynamic payload to a file of this format. We just provide the layer
    // with a default root prim to at least avoid composition errors on the test
    // payload.
    return SdfFileFormat::FindById(
        SdfUsdaFileFormatTokens->Id)->ReadFromString(layer, 
            R"__(#usda 1.0
            (
                defaultPrim = "Root"
            )

            def "Root" {
            }
        )__");
}

// Helper for composing an argument value by name from either a composed 
// metadata field or an attribute default value.
template <class T>
static bool _ComposeArg(const TfToken &argName,
                        const PcpDynamicFileFormatContext& context,
                        T* outValue)
{
    VtValue val;
    if (TfGetEnvSetting(
           TEST_PCP_ASSET_PATH_DYNAMIC_FILE_FORMAT_TOKENS_USE_ATTRIBUTE_INPUTS)) {
        // Attempt to compose the value from an attribute named argName.
        if (!context.ComposeAttributeDefaultValue(argName, &val) || 
                !val.IsHolding<T>()) {
            return false;
        }
    } else {
        // Attempt to compose the value from the prim field named argName.
        if (!context.ComposeValue(argName, &val) || val.IsEmpty()) {
            return false;
        }

        if (!val.IsHolding<T>()) {
            TF_CODING_ERROR(
                "Expected '%s' value to hold an %s, got '%s'", 
                argName.GetText(), 
                TfType::Find<T>().GetTypeName().c_str(),
                TfStringify(val).c_str());
            return false;
        } 
    }

    *outValue = val.UncheckedGet<T>();
    return true;
}

void 
Test_PcpAssetPathDynamicFileFormatPlugin_FileFormat::ComposeFieldsForFileFormatArguments(
    const std::string& assetPath, 
    const PcpDynamicFileFormatContext& context,
    FileFormatArguments* args,
    VtValue *dependencyContextData) const
{
    // Compose the "assetPath" arg.
    SdfAssetPath assetPathArg;
    if (_ComposeArg(Test_PcpAssetPathDynamicFileFormatPlugin_FileFormatTokens->AssetPath, 
                    context, &assetPathArg)) {
        // Add both the resolved and unresolved asset paths as file format 
        // arguments.
        (*args)[Test_PcpAssetPathDynamicFileFormatPlugin_FileFormatTokens->AssetPath] = 
            assetPathArg.GetAssetPath();
        (*args)[Test_PcpAssetPathDynamicFileFormatPlugin_FileFormatTokens->ResolvedAssetPath] = 
            assetPathArg.GetResolvedPath();
    }
    // Compose the "assetPathArray" arg.
    VtArray<SdfAssetPath> assetPathArrayArg;
    if (_ComposeArg(Test_PcpAssetPathDynamicFileFormatPlugin_FileFormatTokens->AssetPathArray, 
                    context, &assetPathArrayArg)) {
        // Add both the list of resolved and list of unresolved asset paths as 
        // file format arguments (converting the lists to string format for 
        // argument purposes).
        std::vector<std::string> assetPaths;
        std::vector<std::string> resolvedAssetPaths;
        for (const auto &assetPathArg : assetPathArrayArg) {
            assetPaths.push_back(assetPathArg.GetAssetPath());
            resolvedAssetPaths.push_back(assetPathArg.GetResolvedPath());
        }
        (*args)[Test_PcpAssetPathDynamicFileFormatPlugin_FileFormatTokens->AssetPathArray] = 
            TfStringPrintf("[%s]", TfStringJoin(assetPaths, ",").c_str());
        (*args)[Test_PcpAssetPathDynamicFileFormatPlugin_FileFormatTokens->ResolvedAssetPathArray] = 
            TfStringPrintf("[%s]", TfStringJoin(resolvedAssetPaths, ",").c_str());
    }
}

PXR_NAMESPACE_CLOSE_SCOPE



