// Copyright (C) 2022-2023 Intel Corporation
// Part of the Unified-Runtime Project, under the Apache License v2.0 with LLVM
// Exceptions. See LICENSE.TXT
//
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception

#include <uur/fixtures.h>

using urDeviceRetainTest = uur::urDeviceTest;
UUR_INSTANTIATE_DEVICE_TEST_SUITE(urDeviceRetainTest);

TEST_P(urDeviceRetainTest, Success) {
  uint32_t prevRefCount = 0;
  ASSERT_SUCCESS(uur::GetObjectReferenceCount(device, prevRefCount));

  ASSERT_SUCCESS(urDeviceRetain(device));

  uint32_t refCount = 0;
  ASSERT_SUCCESS(uur::GetObjectReferenceCount(device, refCount));

  /* If device is a root level device, the device reference counts should
   * remain unchanged */
  ASSERT_EQ(prevRefCount, refCount);
}

TEST_P(urDeviceRetainTest, SuccessSubdevices) {
  if (!uur::hasDevicePartitionSupport(device, UR_DEVICE_PARTITION_EQUALLY)) {
    GTEST_SKIP() << "Device: \'" << device
                 << "\' does not support partitioning equally.";
  }

  ur_device_partition_property_t prop = uur::makePartitionEquallyDesc(1);
  ur_device_partition_properties_t properties{
      UR_STRUCTURE_TYPE_DEVICE_PARTITION_PROPERTIES,
      nullptr,
      &prop,
      1,
  };

  ur_device_handle_t sub_device = nullptr;
  ASSERT_SUCCESS(
      urDevicePartition(device, &properties, 1, &sub_device, nullptr));

  uint32_t prevRefCount = 0;
  ASSERT_SUCCESS(uur::GetObjectReferenceCount(sub_device, prevRefCount));

  ASSERT_SUCCESS(urDeviceRetain(sub_device));

  uint32_t refCount = 0;
  ASSERT_SUCCESS(uur::GetObjectReferenceCount(sub_device, refCount));

  ASSERT_LT(prevRefCount, refCount);

  EXPECT_SUCCESS(urDeviceRelease(sub_device));
  EXPECT_SUCCESS(urDeviceRelease(sub_device));
}

TEST_P(urDeviceRetainTest, InvalidNullHandle) {
  ASSERT_EQ_RESULT(UR_RESULT_ERROR_INVALID_NULL_HANDLE,
                   urDeviceRetain(nullptr));
}
