/** @file
  This is a unit test for RSA OAEP encrypt/decrypt.

  Copyright (c) 2019, Intel Corporation. All rights reserved.<BR>
  Copyright (c) Microsoft Corporation. All rights reserved.
  SPDX-License-Identifier: BSD-2-Clause-Patent
**/

#include "TestBaseCryptLib.h"

STATIC CONST  UINT8  RandSeed[] = "This is the random seed for PRNG verification.";

//
// Self signed X509 certificate
// CN = ca.self
// O = Intel
//
GLOBAL_REMOVE_IF_UNREFERENCED STATIC CONST UINT8  SelfTestCert[] = {
  0x30, 0x82, 0x03, 0x90, 0x30, 0x82, 0x02, 0x78, 0x02, 0x09, 0x00, 0xE4, 0xDF, 0x47, 0x80, 0xEF,
  0x4B, 0x3C, 0x6D, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B,
  0x05, 0x00, 0x30, 0x81, 0x89, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
  0x43, 0x4E, 0x31, 0x11, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0C, 0x08, 0x53, 0x68, 0x61,
  0x6E, 0x67, 0x48, 0x61, 0x69, 0x31, 0x11, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0C, 0x08,
  0x53, 0x68, 0x61, 0x6E, 0x67, 0x48, 0x61, 0x69, 0x31, 0x0E, 0x30, 0x0C, 0x06, 0x03, 0x55, 0x04,
  0x0A, 0x0C, 0x05, 0x49, 0x6E, 0x74, 0x65, 0x6C, 0x31, 0x0C, 0x30, 0x0A, 0x06, 0x03, 0x55, 0x04,
  0x0B, 0x0C, 0x03, 0x46, 0x49, 0x45, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C,
  0x07, 0x63, 0x61, 0x2E, 0x73, 0x65, 0x6C, 0x66, 0x31, 0x24, 0x30, 0x22, 0x06, 0x09, 0x2A, 0x86,
  0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x01, 0x16, 0x15, 0x7A, 0x68, 0x69, 0x63, 0x68, 0x61, 0x6F,
  0x2E, 0x67, 0x61, 0x6F, 0x40, 0x69, 0x6E, 0x74, 0x65, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x30, 0x1E,
  0x17, 0x0D, 0x31, 0x39, 0x30, 0x33, 0x32, 0x37, 0x30, 0x35, 0x35, 0x37, 0x30, 0x39, 0x5A, 0x17,
  0x0D, 0x32, 0x30, 0x30, 0x33, 0x32, 0x36, 0x30, 0x35, 0x35, 0x37, 0x30, 0x39, 0x5A, 0x30, 0x81,
  0x89, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x43, 0x4E, 0x31, 0x11,
  0x30, 0x0F, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0C, 0x08, 0x53, 0x68, 0x61, 0x6E, 0x67, 0x48, 0x61,
  0x69, 0x31, 0x11, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0C, 0x08, 0x53, 0x68, 0x61, 0x6E,
  0x67, 0x48, 0x61, 0x69, 0x31, 0x0E, 0x30, 0x0C, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x05, 0x49,
  0x6E, 0x74, 0x65, 0x6C, 0x31, 0x0C, 0x30, 0x0A, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x03, 0x46,
  0x49, 0x45, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x07, 0x63, 0x61, 0x2E,
  0x73, 0x65, 0x6C, 0x66, 0x31, 0x24, 0x30, 0x22, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D,
  0x01, 0x09, 0x01, 0x16, 0x15, 0x7A, 0x68, 0x69, 0x63, 0x68, 0x61, 0x6F, 0x2E, 0x67, 0x61, 0x6F,
  0x40, 0x69, 0x6E, 0x74, 0x65, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0D,
  0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01,
  0x0F, 0x00, 0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xBC, 0xE4, 0x67, 0xDC, 0xC7,
  0xEA, 0x6F, 0x8A, 0xA7, 0xCC, 0xB2, 0x54, 0x47, 0x48, 0x6A, 0xE2, 0x39, 0xFF, 0xC2, 0x48, 0x58,
  0x34, 0x07, 0x03, 0x6D, 0x39, 0xB3, 0x67, 0x46, 0x4C, 0xBC, 0xA0, 0xFA, 0x4E, 0x64, 0x23, 0x56,
  0x47, 0x7B, 0xC9, 0x1A, 0x2A, 0x55, 0x42, 0x54, 0x10, 0x18, 0x30, 0x92, 0x60, 0x30, 0x5B, 0x9E,
  0xC0, 0x65, 0xD2, 0xD4, 0x05, 0x4A, 0xA6, 0x10, 0x66, 0x04, 0xA9, 0x54, 0x4E, 0xEE, 0x49, 0x39,
  0x43, 0x65, 0x1E, 0x2E, 0x28, 0xDE, 0x79, 0x24, 0xA9, 0x7E, 0xD8, 0x5B, 0xBC, 0x2F, 0x46, 0x6A,
  0xB7, 0xB6, 0x0D, 0x17, 0x88, 0x37, 0x52, 0x5C, 0xFE, 0x93, 0xC0, 0xE2, 0xFD, 0x6A, 0x08, 0x1B,
  0xFB, 0xD1, 0x87, 0xBD, 0xBD, 0x58, 0x57, 0x2C, 0x06, 0x5D, 0xD2, 0x7D, 0x52, 0xE2, 0x49, 0x8E,
  0xDC, 0xE5, 0x26, 0xBD, 0x92, 0x60, 0xB0, 0x3F, 0x58, 0x5E, 0x52, 0xD7, 0x91, 0xDA, 0x93, 0x62,
  0x8D, 0x71, 0x80, 0x53, 0xBA, 0x15, 0xC4, 0x1F, 0xF3, 0xBD, 0xE0, 0xC5, 0xA4, 0xB8, 0xD3, 0x64,
  0x12, 0x14, 0x1B, 0x11, 0x6B, 0x7B, 0xC2, 0x92, 0xC7, 0xE2, 0x94, 0x0B, 0xB8, 0x67, 0x38, 0x48,
  0x63, 0x11, 0x74, 0x25, 0x7C, 0x37, 0xC3, 0xB2, 0xAE, 0xD9, 0xA7, 0x17, 0x9C, 0x4B, 0x9D, 0x6C,
  0x27, 0xB0, 0x87, 0x16, 0x6B, 0xF2, 0x96, 0xE5, 0x1D, 0x37, 0x27, 0xDE, 0xF2, 0x98, 0xB7, 0x81,
  0x08, 0xD9, 0x7A, 0xBA, 0x84, 0x14, 0x61, 0x60, 0x48, 0xCE, 0xCE, 0x51, 0x73, 0xF4, 0xDB, 0xF1,
  0x5F, 0x7A, 0x17, 0x71, 0x4F, 0xC1, 0x0B, 0xCE, 0xC7, 0x31, 0xC1, 0x4E, 0xA3, 0xEE, 0x6F, 0x72,
  0x97, 0x90, 0xFB, 0x8B, 0x54, 0x9F, 0x82, 0x5B, 0x48, 0x5A, 0xF1, 0xAD, 0x8B, 0x3A, 0xCD, 0xCA,
  0xB2, 0x8B, 0x7A, 0x53, 0xD4, 0xF7, 0x71, 0x16, 0x75, 0xA7, 0x35, 0x02, 0x03, 0x01, 0x00, 0x01,
  0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00, 0x03,
  0x82, 0x01, 0x01, 0x00, 0x78, 0x64, 0xCD, 0x6F, 0xC0, 0xF9, 0xE7, 0xAB, 0x19, 0x0B, 0xA4, 0x92,
  0x88, 0xD7, 0xE3, 0xC4, 0x9C, 0xCF, 0xF2, 0x49, 0xFC, 0xCE, 0x8C, 0xEC, 0x33, 0x3F, 0x24, 0x6F,
  0xF7, 0xBC, 0x7C, 0x28, 0xEF, 0xAE, 0xAB, 0x7B, 0xAF, 0x84, 0xE3, 0x4E, 0x38, 0xAB, 0x66, 0x20,
  0x7C, 0xB9, 0xE0, 0x37, 0xFD, 0x84, 0x28, 0x3C, 0xEE, 0x35, 0xC4, 0x75, 0x04, 0x43, 0xCA, 0x4F,
  0x36, 0x60, 0xC4, 0x06, 0x16, 0x54, 0x3C, 0xF9, 0x47, 0x78, 0xBD, 0xE9, 0x7F, 0xF1, 0xE3, 0x0B,
  0xC7, 0x7E, 0x30, 0xA4, 0xCB, 0x3A, 0xC7, 0xCD, 0x50, 0x9A, 0x6C, 0xC1, 0xE3, 0xF8, 0xF8, 0x16,
  0xAA, 0x67, 0x7F, 0x92, 0xEE, 0x1D, 0x47, 0xF2, 0x96, 0xDF, 0xD7, 0x43, 0x64, 0x78, 0xA7, 0xD9,
  0xF6, 0xA1, 0xD9, 0xEB, 0x5B, 0x2E, 0xC6, 0x9E, 0xEA, 0xD5, 0x04, 0x52, 0xA4, 0x73, 0x97, 0xC8,
  0xB3, 0xFD, 0xE1, 0x7E, 0xB1, 0xA4, 0x3F, 0xF4, 0x56, 0x66, 0x8E, 0x6B, 0xBC, 0xD8, 0xF0, 0x7C,
  0xBF, 0x58, 0xA4, 0xE0, 0x45, 0xDC, 0x2B, 0x0B, 0x2C, 0x40, 0x52, 0x4D, 0x4B, 0xA9, 0x8A, 0x77,
  0x87, 0x31, 0xF8, 0x45, 0xD9, 0x60, 0xBB, 0xAD, 0x28, 0x29, 0x14, 0x0D, 0x0D, 0x8F, 0x0F, 0x34,
  0x76, 0x88, 0xF3, 0x5A, 0x65, 0x00, 0x54, 0x22, 0x35, 0xBF, 0x3A, 0x2B, 0xE9, 0xB0, 0x24, 0xE5,
  0xBE, 0xDF, 0x08, 0xA0, 0x02, 0xB2, 0x81, 0x01, 0xE8, 0x14, 0xBE, 0x6E, 0x7E, 0xBD, 0x60, 0x0C,
  0x4A, 0xD8, 0x57, 0xA4, 0x50, 0xD3, 0xD0, 0x59, 0xCE, 0x57, 0x27, 0xE9, 0xA6, 0x16, 0x15, 0x5F,
  0xF2, 0x6C, 0xCC, 0x57, 0x36, 0x20, 0x06, 0xE6, 0xA7, 0x24, 0x22, 0xB0, 0xB1, 0xC4, 0xBC, 0x2C,
  0x5C, 0xCE, 0x92, 0x11, 0xFF, 0x6F, 0x8C, 0x14, 0x7E, 0xC4, 0x10, 0x02, 0xF6, 0x69, 0x86, 0x6D,
  0x5B, 0x64, 0x81, 0x13,
};

GLOBAL_REMOVE_IF_UNREFERENCED STATIC CONST UINT8  PrivateKey[] = {
  0x30, 0x82, 0x04, 0xA4, 0x02, 0x01, 0x00, 0x02, 0x82, 0x01, 0x01, 0x00, 0xBC, 0xE4, 0x67, 0xDC,
  0xC7, 0xEA, 0x6F, 0x8A, 0xA7, 0xCC, 0xB2, 0x54, 0x47, 0x48, 0x6A, 0xE2, 0x39, 0xFF, 0xC2, 0x48,
  0x58, 0x34, 0x07, 0x03, 0x6D, 0x39, 0xB3, 0x67, 0x46, 0x4C, 0xBC, 0xA0, 0xFA, 0x4E, 0x64, 0x23,
  0x56, 0x47, 0x7B, 0xC9, 0x1A, 0x2A, 0x55, 0x42, 0x54, 0x10, 0x18, 0x30, 0x92, 0x60, 0x30, 0x5B,
  0x9E, 0xC0, 0x65, 0xD2, 0xD4, 0x05, 0x4A, 0xA6, 0x10, 0x66, 0x04, 0xA9, 0x54, 0x4E, 0xEE, 0x49,
  0x39, 0x43, 0x65, 0x1E, 0x2E, 0x28, 0xDE, 0x79, 0x24, 0xA9, 0x7E, 0xD8, 0x5B, 0xBC, 0x2F, 0x46,
  0x6A, 0xB7, 0xB6, 0x0D, 0x17, 0x88, 0x37, 0x52, 0x5C, 0xFE, 0x93, 0xC0, 0xE2, 0xFD, 0x6A, 0x08,
  0x1B, 0xFB, 0xD1, 0x87, 0xBD, 0xBD, 0x58, 0x57, 0x2C, 0x06, 0x5D, 0xD2, 0x7D, 0x52, 0xE2, 0x49,
  0x8E, 0xDC, 0xE5, 0x26, 0xBD, 0x92, 0x60, 0xB0, 0x3F, 0x58, 0x5E, 0x52, 0xD7, 0x91, 0xDA, 0x93,
  0x62, 0x8D, 0x71, 0x80, 0x53, 0xBA, 0x15, 0xC4, 0x1F, 0xF3, 0xBD, 0xE0, 0xC5, 0xA4, 0xB8, 0xD3,
  0x64, 0x12, 0x14, 0x1B, 0x11, 0x6B, 0x7B, 0xC2, 0x92, 0xC7, 0xE2, 0x94, 0x0B, 0xB8, 0x67, 0x38,
  0x48, 0x63, 0x11, 0x74, 0x25, 0x7C, 0x37, 0xC3, 0xB2, 0xAE, 0xD9, 0xA7, 0x17, 0x9C, 0x4B, 0x9D,
  0x6C, 0x27, 0xB0, 0x87, 0x16, 0x6B, 0xF2, 0x96, 0xE5, 0x1D, 0x37, 0x27, 0xDE, 0xF2, 0x98, 0xB7,
  0x81, 0x08, 0xD9, 0x7A, 0xBA, 0x84, 0x14, 0x61, 0x60, 0x48, 0xCE, 0xCE, 0x51, 0x73, 0xF4, 0xDB,
  0xF1, 0x5F, 0x7A, 0x17, 0x71, 0x4F, 0xC1, 0x0B, 0xCE, 0xC7, 0x31, 0xC1, 0x4E, 0xA3, 0xEE, 0x6F,
  0x72, 0x97, 0x90, 0xFB, 0x8B, 0x54, 0x9F, 0x82, 0x5B, 0x48, 0x5A, 0xF1, 0xAD, 0x8B, 0x3A, 0xCD,
  0xCA, 0xB2, 0x8B, 0x7A, 0x53, 0xD4, 0xF7, 0x71, 0x16, 0x75, 0xA7, 0x35, 0x02, 0x03, 0x01, 0x00,
  0x01, 0x02, 0x82, 0x01, 0x00, 0x13, 0xF7, 0xD1, 0x42, 0xF5, 0x9F, 0x42, 0xCB, 0x55, 0x91, 0xBE,
  0x08, 0x4A, 0xC0, 0xCD, 0x0B, 0xBD, 0x35, 0xDC, 0x43, 0xE9, 0x8F, 0x16, 0x6E, 0xB6, 0x4D, 0x33,
  0x39, 0xE7, 0xA4, 0x95, 0x0C, 0x2F, 0x69, 0xBA, 0x0C, 0x42, 0x42, 0xAC, 0x43, 0x46, 0x10, 0xD3,
  0x92, 0x7F, 0x70, 0x74, 0x1E, 0x2E, 0x5B, 0x1C, 0xC1, 0x92, 0xB6, 0xA4, 0x0C, 0xF5, 0x7C, 0xD9,
  0xB7, 0x54, 0x64, 0x74, 0x79, 0xB1, 0xFF, 0xE6, 0x10, 0xB7, 0x8C, 0xF8, 0x53, 0x88, 0x6D, 0xA9,
  0x97, 0x04, 0xD9, 0x26, 0x1F, 0x99, 0x12, 0xFB, 0xAC, 0x65, 0xFB, 0xA5, 0xB3, 0x1C, 0x99, 0xB9,
  0xBF, 0x6B, 0x35, 0x3E, 0x49, 0x55, 0xB5, 0x94, 0x4F, 0xE7, 0x25, 0x67, 0xB1, 0x01, 0xCD, 0xD2,
  0x58, 0xE4, 0xBE, 0x87, 0x8C, 0x88, 0xD3, 0x0A, 0x38, 0xDC, 0x71, 0x5D, 0x88, 0x0A, 0xE2, 0x3E,
  0x76, 0x63, 0x3B, 0xE4, 0x3C, 0x8F, 0x2F, 0x29, 0x1D, 0xD1, 0x66, 0x8D, 0xC0, 0x4A, 0x68, 0x15,
  0x90, 0x4C, 0x95, 0x61, 0xF4, 0xFD, 0xE8, 0xFA, 0x9C, 0x6C, 0x00, 0x22, 0x23, 0xD5, 0x17, 0x6E,
  0xEE, 0xA8, 0xD8, 0x70, 0xC5, 0x74, 0xEA, 0x09, 0x13, 0x7F, 0x0C, 0x37, 0x4D, 0x50, 0xCD, 0xE9,
  0x16, 0xC2, 0xD5, 0xDE, 0x5E, 0xC3, 0xFC, 0x46, 0x08, 0xF1, 0x99, 0xC0, 0xB4, 0x28, 0xFD, 0x2B,
  0x29, 0xEF, 0x76, 0xD7, 0x04, 0x4F, 0x02, 0x54, 0x16, 0x54, 0x55, 0x20, 0xEC, 0xBC, 0xBF, 0x85,
  0x5F, 0x12, 0xCC, 0xFC, 0x0D, 0xF2, 0xEF, 0xFC, 0x4D, 0x3E, 0xA2, 0x5E, 0x97, 0xFE, 0x35, 0x10,
  0x0F, 0x53, 0x1F, 0x80, 0xD5, 0xC0, 0xB4, 0xE9, 0xE9, 0x31, 0x4C, 0x89, 0x14, 0x72, 0x39, 0x65,
  0x89, 0xEF, 0x7A, 0x51, 0x4A, 0xB9, 0xA9, 0xCC, 0x1B, 0x52, 0xB0, 0x02, 0x52, 0x65, 0x2F, 0x0B,
  0x89, 0x41, 0x70, 0x1E, 0x01, 0x02, 0x81, 0x81, 0x00, 0xDB, 0xE8, 0x21, 0x9E, 0xDB, 0x58, 0x42,
  0x0B, 0x02, 0x35, 0x87, 0xDF, 0x35, 0xF4, 0xC1, 0xEE, 0xD5, 0x72, 0x2C, 0xDC, 0x4D, 0xC7, 0xDA,
  0x6D, 0x18, 0x0F, 0x68, 0x6F, 0xE7, 0xB0, 0x6C, 0xA0, 0xB2, 0x9E, 0xB8, 0x5E, 0x4A, 0x86, 0x81,
  0x97, 0xBD, 0x78, 0xFC, 0x06, 0xD5, 0xBE, 0x98, 0x42, 0x38, 0x78, 0x78, 0x16, 0xAE, 0x39, 0x56,
  0xFB, 0x67, 0x6B, 0xED, 0x7F, 0xB3, 0xC3, 0x7D, 0x95, 0x86, 0x05, 0x0C, 0xF1, 0x4C, 0xC2, 0x67,
  0x94, 0x93, 0x1B, 0xB7, 0x91, 0x0A, 0x80, 0xE2, 0x85, 0xA7, 0x75, 0x20, 0x61, 0xCD, 0x40, 0x5B,
  0x39, 0x13, 0x32, 0xF7, 0xB8, 0xD7, 0x0C, 0x39, 0x4C, 0x82, 0x4B, 0xA3, 0x9F, 0x5E, 0xB1, 0xB0,
  0x38, 0x81, 0x8B, 0x1E, 0xBD, 0x2B, 0xC2, 0x9F, 0x3A, 0x68, 0x72, 0xC2, 0x90, 0xCB, 0xA2, 0xE2,
  0xAC, 0x86, 0xFF, 0xC8, 0xD7, 0xB8, 0x97, 0x0E, 0x69, 0x02, 0x81, 0x81, 0x00, 0xDB, 0xE5, 0x21,
  0x1F, 0x4C, 0x83, 0xD9, 0x7A, 0x56, 0x87, 0x82, 0xF1, 0x40, 0x1A, 0xD3, 0x5B, 0xC5, 0xB4, 0x4D,
  0x18, 0x8A, 0xE3, 0x6B, 0xC9, 0x7E, 0x8F, 0x5D, 0x00, 0x33, 0x66, 0xEB, 0xAE, 0x8E, 0x54, 0x49,
  0xD7, 0xEA, 0xF3, 0xF9, 0xED, 0xD7, 0xEE, 0x2C, 0xC8, 0xD6, 0x8F, 0x40, 0x0A, 0x53, 0xCB, 0x5D,
  0x16, 0xB7, 0xEB, 0x4E, 0xFB, 0x11, 0x18, 0x1B, 0x4B, 0x6B, 0x43, 0x04, 0xD6, 0x1D, 0x8F, 0x6D,
  0x4D, 0x48, 0x8B, 0x6C, 0x82, 0x52, 0xB4, 0x8D, 0x58, 0x9B, 0x19, 0x8D, 0xE0, 0x0D, 0xFD, 0x15,
  0x9D, 0x35, 0xDF, 0x5E, 0xED, 0xDB, 0x8F, 0x8C, 0x44, 0x11, 0xC9, 0x9A, 0xD9, 0x3F, 0xD4, 0xA5,
  0xEE, 0x68, 0x3B, 0x38, 0xD1, 0x66, 0xB0, 0x62, 0x7E, 0xA5, 0x04, 0xD1, 0x67, 0xA4, 0x1D, 0x93,
  0xD6, 0x2A, 0xFB, 0xF6, 0xC1, 0xB2, 0xC4, 0x02, 0x8A, 0xB2, 0xA8, 0xD0, 0xED, 0x02, 0x81, 0x81,
  0x00, 0xD1, 0x1D, 0x98, 0x43, 0x09, 0x10, 0x8F, 0x95, 0x0A, 0x5F, 0xBF, 0x3A, 0x1A, 0x84, 0x0A,
  0x9B, 0x44, 0x6C, 0x53, 0x8E, 0x97, 0x84, 0x72, 0x08, 0x85, 0xF5, 0xB8, 0xF9, 0x99, 0x58, 0x22,
  0xA4, 0x59, 0xD7, 0x27, 0x39, 0xBB, 0x22, 0xD7, 0x3F, 0x65, 0x95, 0x14, 0x0C, 0xA7, 0xDB, 0x9B,
  0xCC, 0x75, 0x08, 0xD2, 0x21, 0xB7, 0x41, 0x9D, 0x9E, 0x6A, 0x85, 0xAE, 0x8E, 0x5E, 0x72, 0x8B,
  0x0C, 0xD2, 0x2A, 0x21, 0x59, 0xA1, 0x05, 0xC9, 0x1B, 0xF7, 0x1C, 0x47, 0x5E, 0x6C, 0x9F, 0x8D,
  0x49, 0x4D, 0xA9, 0xCA, 0xC6, 0x56, 0x39, 0x59, 0xC6, 0x17, 0xA4, 0xC2, 0xFF, 0x09, 0xB8, 0x7D,
  0x10, 0x6D, 0x62, 0xE2, 0x50, 0x9D, 0xEA, 0xD8, 0xC9, 0x93, 0x1E, 0xA9, 0x77, 0x78, 0x92, 0x95,
  0x7A, 0x24, 0x72, 0xE4, 0xE4, 0x6E, 0x51, 0x11, 0xBE, 0xC0, 0x8D, 0xA1, 0xFE, 0xC9, 0x71, 0x13,
  0xE9, 0x02, 0x81, 0x80, 0x38, 0xE5, 0x3F, 0x3D, 0xB6, 0xF1, 0x82, 0x8A, 0xD9, 0xEA, 0xB4, 0x21,
  0xB1, 0xE1, 0x7F, 0x00, 0x86, 0xD6, 0x11, 0x00, 0xA1, 0xA7, 0x18, 0x53, 0x89, 0x50, 0x94, 0xD2,
  0xA1, 0x7A, 0x26, 0xCF, 0x02, 0x7A, 0xA2, 0xA4, 0xA0, 0x61, 0x42, 0x30, 0x9D, 0x27, 0x3E, 0x25,
  0x35, 0x06, 0xD9, 0x9E, 0x12, 0x72, 0x53, 0x02, 0xB6, 0xD1, 0xC6, 0x9C, 0xC5, 0x9E, 0x84, 0x29,
  0x6E, 0x76, 0x67, 0xC0, 0xF2, 0xE3, 0x12, 0x77, 0xE5, 0xA6, 0x55, 0x8C, 0x1E, 0xD4, 0x22, 0xA2,
  0x63, 0x77, 0x28, 0x33, 0x69, 0xCB, 0x2B, 0x15, 0x83, 0x43, 0xE9, 0x6F, 0x07, 0xD9, 0x4B, 0x84,
  0xAD, 0x49, 0x89, 0xAC, 0x5E, 0x8C, 0xA9, 0xE6, 0x56, 0xC7, 0xC3, 0xFA, 0xC2, 0x64, 0x2F, 0x07,
  0xEF, 0xB0, 0xA0, 0x71, 0xFD, 0x8F, 0xBD, 0x99, 0xB3, 0x83, 0x80, 0x24, 0x3C, 0x19, 0xE5, 0x42,
  0x77, 0xD9, 0xA0, 0x3D, 0x02, 0x81, 0x81, 0x00, 0xB7, 0x35, 0xE9, 0x66, 0x4A, 0xCD, 0x5F, 0x5C,
  0x44, 0x9A, 0x7F, 0x23, 0xAE, 0xC2, 0x7F, 0xF0, 0x8D, 0xE4, 0xDB, 0xCD, 0x1A, 0x8A, 0x40, 0xD7,
  0x0A, 0xC4, 0x3E, 0xC5, 0x03, 0xFB, 0x47, 0xAB, 0x72, 0xDF, 0xD7, 0x68, 0x27, 0x6E, 0x86, 0x94,
  0x2A, 0x2A, 0xDA, 0x25, 0x9E, 0xF6, 0x5E, 0xE0, 0x6D, 0x7C, 0x7D, 0xFE, 0x72, 0xDD, 0xC7, 0xBC,
  0x74, 0xC7, 0xAD, 0xEA, 0x44, 0xBE, 0x5C, 0x8A, 0x03, 0xCD, 0x35, 0x74, 0x6F, 0x4E, 0x92, 0x31,
  0xBE, 0x57, 0x9A, 0x02, 0x0C, 0xFC, 0xD1, 0xD5, 0xE7, 0x3E, 0x53, 0x0C, 0x47, 0x12, 0x39, 0x45,
  0x6E, 0x35, 0x24, 0xA6, 0xDA, 0x42, 0x63, 0xC6, 0xBF, 0xCA, 0xE8, 0x32, 0xFD, 0x61, 0xC0, 0x80,
  0x39, 0xD2, 0xB7, 0x0D, 0xF3, 0xA2, 0xF3, 0xE4, 0x49, 0x26, 0x89, 0xD0, 0xA0, 0x24, 0xC8, 0x27,
  0x86, 0x10, 0x09, 0x88, 0x6C, 0x35, 0x60, 0xF2,
};

// The following RSA key componets were extracted from the above private key with openssl.
GLOBAL_REMOVE_IF_UNREFERENCED STATIC CONST UINT8  RsaN[] = {
  0x00,
  0xbc,0xe4,  0x67, 0xdc, 0xc7, 0xea, 0x6f, 0x8a, 0xa7, 0xcc, 0xb2, 0x54, 0x47, 0x48, 0x6a, 0xe2,
  0x39,0xff,  0xc2, 0x48, 0x58, 0x34, 0x07, 0x03, 0x6d, 0x39, 0xb3, 0x67, 0x46, 0x4c, 0xbc, 0xa0,
  0xfa,0x4e,  0x64, 0x23, 0x56, 0x47, 0x7b, 0xc9, 0x1a, 0x2a, 0x55, 0x42, 0x54, 0x10, 0x18, 0x30,
  0x92,0x60,  0x30, 0x5b, 0x9e, 0xc0, 0x65, 0xd2, 0xd4, 0x05, 0x4a, 0xa6, 0x10, 0x66, 0x04, 0xa9,
  0x54,0x4e,  0xee, 0x49, 0x39, 0x43, 0x65, 0x1e, 0x2e, 0x28, 0xde, 0x79, 0x24, 0xa9, 0x7e, 0xd8,
  0x5b,0xbc,  0x2f, 0x46, 0x6a, 0xb7, 0xb6, 0x0d, 0x17, 0x88, 0x37, 0x52, 0x5c, 0xfe, 0x93, 0xc0,
  0xe2,0xfd,  0x6a, 0x08, 0x1b, 0xfb, 0xd1, 0x87, 0xbd, 0xbd, 0x58, 0x57, 0x2c, 0x06, 0x5d, 0xd2,
  0x7d,0x52,  0xe2, 0x49, 0x8e, 0xdc, 0xe5, 0x26, 0xbd, 0x92, 0x60, 0xb0, 0x3f, 0x58, 0x5e, 0x52,
  0xd7,0x91,  0xda, 0x93, 0x62, 0x8d, 0x71, 0x80, 0x53, 0xba, 0x15, 0xc4, 0x1f, 0xf3, 0xbd, 0xe0,
  0xc5,0xa4,  0xb8, 0xd3, 0x64, 0x12, 0x14, 0x1b, 0x11, 0x6b, 0x7b, 0xc2, 0x92, 0xc7, 0xe2, 0x94,
  0x0b,0xb8,  0x67, 0x38, 0x48, 0x63, 0x11, 0x74, 0x25, 0x7c, 0x37, 0xc3, 0xb2, 0xae, 0xd9, 0xa7,
  0x17,0x9c,  0x4b, 0x9d, 0x6c, 0x27, 0xb0, 0x87, 0x16, 0x6b, 0xf2, 0x96, 0xe5, 0x1d, 0x37, 0x27,
  0xde,0xf2,  0x98, 0xb7, 0x81, 0x08, 0xd9, 0x7a, 0xba, 0x84, 0x14, 0x61, 0x60, 0x48, 0xce, 0xce,
  0x51,0x73,  0xf4, 0xdb, 0xf1, 0x5f, 0x7a, 0x17, 0x71, 0x4f, 0xc1, 0x0b, 0xce, 0xc7, 0x31, 0xc1,
  0x4e,0xa3,  0xee, 0x6f, 0x72, 0x97, 0x90, 0xfb, 0x8b, 0x54, 0x9f, 0x82, 0x5b, 0x48, 0x5a, 0xf1,
  0xad,0x8b,  0x3a, 0xcd, 0xca, 0xb2, 0x8b, 0x7a, 0x53, 0xd4, 0xf7, 0x71, 0x16, 0x75, 0xa7, 0x35,
};

GLOBAL_REMOVE_IF_UNREFERENCED STATIC CONST UINT8  RsaE[] = {
  0x01, 0x00, 0x01
};

GLOBAL_REMOVE_IF_UNREFERENCED STATIC CONST UINT8  RsaD[] = {
  0x13, 0xf7, 0xd1, 0x42, 0xf5, 0x9f, 0x42, 0xcb, 0x55, 0x91, 0xbe, 0x08, 0x4a, 0xc0, 0xcd, 0x0b,
  0xbd, 0x35, 0xdc, 0x43, 0xe9, 0x8f, 0x16, 0x6e, 0xb6, 0x4d, 0x33, 0x39, 0xe7, 0xa4, 0x95, 0x0c,
  0x2f, 0x69, 0xba, 0x0c, 0x42, 0x42, 0xac, 0x43, 0x46, 0x10, 0xd3, 0x92, 0x7f, 0x70, 0x74, 0x1e,
  0x2e, 0x5b, 0x1c, 0xc1, 0x92, 0xb6, 0xa4, 0x0c, 0xf5, 0x7c, 0xd9, 0xb7, 0x54, 0x64, 0x74, 0x79,
  0xb1, 0xff, 0xe6, 0x10, 0xb7, 0x8c, 0xf8, 0x53, 0x88, 0x6d, 0xa9, 0x97, 0x04, 0xd9, 0x26, 0x1f,
  0x99, 0x12, 0xfb, 0xac, 0x65, 0xfb, 0xa5, 0xb3, 0x1c, 0x99, 0xb9, 0xbf, 0x6b, 0x35, 0x3e, 0x49,
  0x55, 0xb5, 0x94, 0x4f, 0xe7, 0x25, 0x67, 0xb1, 0x01, 0xcd, 0xd2, 0x58, 0xe4, 0xbe, 0x87, 0x8c,
  0x88, 0xd3, 0x0a, 0x38, 0xdc, 0x71, 0x5d, 0x88, 0x0a, 0xe2, 0x3e, 0x76, 0x63, 0x3b, 0xe4, 0x3c,
  0x8f, 0x2f, 0x29, 0x1d, 0xd1, 0x66, 0x8d, 0xc0, 0x4a, 0x68, 0x15, 0x90, 0x4c, 0x95, 0x61, 0xf4,
  0xfd, 0xe8, 0xfa, 0x9c, 0x6c, 0x00, 0x22, 0x23, 0xd5, 0x17, 0x6e, 0xee, 0xa8, 0xd8, 0x70, 0xc5,
  0x74, 0xea, 0x09, 0x13, 0x7f, 0x0c, 0x37, 0x4d, 0x50, 0xcd, 0xe9, 0x16, 0xc2, 0xd5, 0xde, 0x5e,
  0xc3, 0xfc, 0x46, 0x08, 0xf1, 0x99, 0xc0, 0xb4, 0x28, 0xfd, 0x2b, 0x29, 0xef, 0x76, 0xd7, 0x04,
  0x4f, 0x02, 0x54, 0x16, 0x54, 0x55, 0x20, 0xec, 0xbc, 0xbf, 0x85, 0x5f, 0x12, 0xcc, 0xfc, 0x0d,
  0xf2, 0xef, 0xfc, 0x4d, 0x3e, 0xa2, 0x5e, 0x97, 0xfe, 0x35, 0x10, 0x0f, 0x53, 0x1f, 0x80, 0xd5,
  0xc0, 0xb4, 0xe9, 0xe9, 0x31, 0x4c, 0x89, 0x14, 0x72, 0x39, 0x65, 0x89, 0xef, 0x7a, 0x51, 0x4a,
  0xb9, 0xa9, 0xcc, 0x1b, 0x52, 0xb0, 0x02, 0x52, 0x65, 0x2f, 0x0b, 0x89, 0x41, 0x70, 0x1e, 0x01,
};

// test case = "123\0"
GLOBAL_REMOVE_IF_UNREFERENCED STATIC CONST UINT8  Msg1230[] = {
  0x31, 0x32, 0x33, 0x00
};

// Ciphertext of the test case using RSAES-OAEP2048 with SHA1 MD/BGF1 created with openssl.
GLOBAL_REMOVE_IF_UNREFERENCED STATIC CONST UINT8  Ct1230RsaesOaepMdSha1Mgf1Sha1[] = {
  0x88, 0x5d, 0xf3, 0x00, 0x66, 0x77, 0x91, 0x94, 0x5c, 0x8d, 0x45, 0xb6, 0xb2, 0x24, 0x26, 0x26,
  0x37, 0xbe, 0xe0, 0x87, 0x4f, 0x50, 0xbf, 0x88, 0xde, 0x5d, 0xe9, 0xe0, 0xb2, 0x7e, 0x66, 0xfa,
  0x6c, 0xfd, 0x0d, 0x19, 0x48, 0x41, 0xfe, 0x7a, 0x86, 0xa8, 0x28, 0xc2, 0x01, 0xcf, 0x76, 0xd7,
  0xea, 0xab, 0x6d, 0xc3, 0x5e, 0x2c, 0x36, 0x04, 0xc0, 0x54, 0xc2, 0x68, 0x67, 0xe7, 0x04, 0x27,
  0x56, 0xbe, 0x53, 0xb5, 0x80, 0x94, 0xd8, 0xde, 0x8c, 0x75, 0x69, 0x42, 0xba, 0x55, 0xd6, 0x2c,
  0xda, 0x22, 0xe6, 0x09, 0xf6, 0x90, 0x27, 0x4b, 0x10, 0x54, 0x40, 0xa0, 0x74, 0x31, 0xdb, 0x5f,
  0x80, 0x06, 0xc7, 0x67, 0x96, 0xe8, 0x45, 0xea, 0x7f, 0x72, 0x18, 0x24, 0xe8, 0x0d, 0x46, 0xc2,
  0xa0, 0x83, 0xca, 0x71, 0xca, 0x91, 0x4b, 0x89, 0x80, 0x61, 0x01, 0x8e, 0xcf, 0xa1, 0x68, 0x81,
  0x2d, 0xf2, 0x08, 0xd2, 0x02, 0x9e, 0xc0, 0xa4, 0x91, 0x71, 0x90, 0x84, 0x2f, 0x4e, 0x18, 0x37,
  0x9b, 0x61, 0x0b, 0xf5, 0x88, 0xf7, 0x6b, 0x87, 0xb9, 0x4e, 0x31, 0xda, 0xf3, 0xb5, 0xe2, 0x60,
  0x4d, 0xd9, 0x52, 0x99, 0x6b, 0x19, 0x98, 0xa2, 0x28, 0xaa, 0xeb, 0x5a, 0x33, 0xef, 0xf1, 0x4e,
  0x29, 0x86, 0xbf, 0x70, 0x08, 0xfd, 0x34, 0x8a, 0x8c, 0x6d, 0xef, 0xc4, 0xa1, 0xfe, 0xdf, 0x4d,
  0xeb, 0xf0, 0x2c, 0x4c, 0xf5, 0xb3, 0xe8, 0xf8, 0xc3, 0x45, 0xc7, 0x6b, 0x59, 0x1c, 0x9b, 0xd9,
  0x52, 0xdf, 0x65, 0x87, 0x18, 0xd2, 0x6d, 0xff, 0x8b, 0x98, 0x2a, 0x97, 0xeb, 0x93, 0xea, 0x6a,
  0x23, 0x23, 0xc6, 0x32, 0xf5, 0xea, 0x45, 0xe3, 0x99, 0xa0, 0x4d, 0x4b, 0x8f, 0xf8, 0x1d, 0xad,
  0xa9, 0x97, 0xa2, 0xd6, 0xaf, 0x5e, 0x11, 0xf7, 0x5f, 0x28, 0xfb, 0x38, 0x80, 0x38, 0x50, 0xc4,
};

// Ciphertext of the test case using RSAES-OAEP2048 with SHA256 MD/BGF1 created with openssl.
GLOBAL_REMOVE_IF_UNREFERENCED STATIC CONST UINT8  Ct1230RsaesOaep2048MdSha256Mgf1Sha256[] = {
  0xa7, 0x20, 0xa9, 0x31, 0xb5, 0xad, 0x83, 0x0a, 0x07, 0xee, 0x36, 0x46, 0xa5, 0x78, 0x3a, 0xda,
  0x9d, 0xdf, 0xe6, 0x05, 0x0f, 0x7c, 0x46, 0xfe, 0x5f, 0xd6, 0x58, 0x16, 0xb6, 0xaa, 0x82, 0x7c,
  0x58, 0x8a, 0x52, 0x14, 0x12, 0x29, 0x6f, 0x62, 0x80, 0xa7, 0x61, 0xfe, 0x29, 0x72, 0x6f, 0x73,
  0xf6, 0x2f, 0x54, 0x38, 0x58, 0x7b, 0xbd, 0xa1, 0x2f, 0x9d, 0x12, 0x83, 0x72, 0xbc, 0x3d, 0x29,
  0x65, 0x39, 0xcb, 0x93, 0x95, 0x3e, 0x73, 0xc9, 0x6f, 0xb9, 0xe8, 0xd5, 0x8b, 0x91, 0x0d, 0x87,
  0x7e, 0x22, 0xb5, 0x93, 0x3d, 0xa8, 0x4a, 0xd9, 0x1a, 0x13, 0xf7, 0xf4, 0x7f, 0x16, 0x42, 0xfe,
  0x63, 0x10, 0x7e, 0xa1, 0xe5, 0x04, 0xcf, 0xed, 0x93, 0x2d, 0x16, 0x3b, 0x79, 0x1f, 0x53, 0x41,
  0xe3, 0xca, 0x69, 0x18, 0x6a, 0xe5, 0xec, 0x9a, 0xce, 0xbc, 0x47, 0xf6, 0x77, 0x9a, 0x5c, 0xea,
  0xac, 0x7e, 0x28, 0xeb, 0x1e, 0xfe, 0x75, 0xa6, 0xbf, 0x1e, 0xfd, 0x1c, 0x63, 0x69, 0x47, 0x04,
  0xaf, 0x69, 0x7e, 0x1c, 0xa1, 0x7f, 0x00, 0xcf, 0xec, 0x16, 0x34, 0xd9, 0xde, 0x91, 0x0e, 0x0f,
  0x0b, 0x1e, 0x66, 0xc3, 0x41, 0x88, 0x43, 0xbe, 0xa3, 0x2a, 0x7c, 0x87, 0xff, 0xc0, 0x67, 0xdc,
  0xc7, 0xeb, 0x28, 0x07, 0x00, 0x72, 0x85, 0x17, 0xca, 0x05, 0x9f, 0x29, 0x6b, 0xad, 0xc6, 0xae,
  0x1c, 0x4a, 0xf2, 0xfe, 0x97, 0xc7, 0x6e, 0x4b, 0xbf, 0xfd, 0x46, 0xbe, 0xf8, 0x76, 0xc9, 0x70,
  0x58, 0x3a, 0x73, 0xcc, 0x34, 0xda, 0xfe, 0x5b, 0x6d, 0x98, 0x74, 0x95, 0x85, 0xc7, 0xc9, 0x84,
  0x02, 0xa8, 0x97, 0x13, 0xa3, 0x83, 0xcb, 0x28, 0x3d, 0xbb, 0x2b, 0x3b, 0x45, 0xf1, 0x6e, 0xc5,
  0x37, 0x23, 0x21, 0xe6, 0x74, 0x2d, 0x48, 0x19, 0x97, 0xaf, 0xee, 0x3d, 0x9b, 0xd0, 0x05, 0xc7
};

typedef struct _OAEP_ENC_DEC_TEST_CONTEXT OAEP_ENC_DEC_TEST_CONTEXT;
typedef
BOOLEAN
(EFIAPI *OAEP_TEST_ENCRYPT)(
  IN      OAEP_ENC_DEC_TEST_CONTEXT *TestContext,
  IN      CONST UINT8 *ClearText,
  IN      UINTN       ClearTextSize,
  IN      CONST UINT8 *PrngSeed,
  IN      UINTN       PrngSeedSize,
  IN      UINT16      DigestLen,
  OUT     UINT8       **CipherText,
  OUT     UINTN       *CipherTextSize
  );

typedef
BOOLEAN
(EFIAPI *OAEP_TEST_DECRYPT)(
  IN      OAEP_ENC_DEC_TEST_CONTEXT *TestContext,
  IN      CONST UINT8 *CipherText,
  IN      UINTN       CipherTextSize,
  IN      UINT16      DigestLen,
  OUT     UINT8       **ClearText,
  OUT     UINTN       *ClearTextSize
  );

typedef struct _OAEP_ENC_DEC_TEST_CONTEXT {
  CONST UINT8          *SelfTestCert;
  UINTN                SelfTestCertSize;
  CONST UINT8          *PrivateKey;
  UINTN                PrivateKeySize;
  CONST UINT8          *RsaN;
  UINTN                RsaNSize;
  CONST UINT8          *RsaE;
  UINTN                RsaESize;
  CONST UINT8          *RsaD;
  UINTN                RsaDSize;
  CONST UINT8          *PrngSeed;
  UINTN                PrngSeedSize;
  CONST UINT8          *ClearText;
  UINTN                ClearTextSize;
  CONST UINT8          *CipherText;
  UINTN                CipherTextSize;
  UINT16               DigestLen;
  OAEP_TEST_ENCRYPT    Encrypt;
  OAEP_TEST_DECRYPT    Decrypt;
  UNIT_TEST_STATUS     Expect;
} OAEP_ENC_DEC_TEST_CONTEXT;

BOOLEAN
EFIAPI
CallPkcs1v2Encrypt (
  OAEP_ENC_DEC_TEST_CONTEXT  *TestCtx,
  CONST UINT8                *ClearText,
  UINTN                      ClearTextSize,
  CONST UINT8                *PrngSeed,
  UINTN                      PrngSeedSize,
  UINT16                     DigestLen,
  UINT8                      **CipherText,
  UINTN                      *CipherTextSize
  )
{
  BOOLEAN  Status;

  Status = Pkcs1v2Encrypt (
             TestCtx->SelfTestCert,
             TestCtx->SelfTestCertSize,
             (UINT8 *)ClearText,
             ClearTextSize,
             PrngSeed,
             PrngSeedSize,
             CipherText,
             CipherTextSize
             );

  return Status;
}

BOOLEAN
EFIAPI
CallPkcs1v2Decrypt (
  OAEP_ENC_DEC_TEST_CONTEXT  *TestCtx,
  CONST UINT8                *CipherText,
  UINTN                      CipherTextSize,
  UINT16                     DigestLen,
  UINT8                      **ClearText,
  UINTN                      *ClearTextSize
  )
{
  BOOLEAN  Status;

  Status = Pkcs1v2Decrypt (
             TestCtx->PrivateKey,
             TestCtx->PrivateKeySize,
             (UINT8 *)CipherText,
             CipherTextSize,
             ClearText,
             ClearTextSize
             );
  return Status;
}

BOOLEAN
EFIAPI
CallRsaOaepEncrypt (
  OAEP_ENC_DEC_TEST_CONTEXT  *TestCtx,
  CONST UINT8                *ClearText,
  UINTN                      ClearTextSize,
  CONST UINT8                *RandSeedIn,
  UINTN                      RandSeedSizeIn,
  UINT16                     DigestLen,
  UINT8                      **CipherText,
  UINTN                      *CipherTextSize
  )
{
  VOID     *RsaContext = NULL;
  BOOLEAN  Status;

  RsaContext = RsaNew ();
  UT_ASSERT_FALSE (RsaContext == NULL);

  Status = RsaSetKey (RsaContext, RsaKeyN, TestCtx->RsaN, TestCtx->RsaNSize);
  UT_ASSERT_TRUE (Status);

  Status = RsaSetKey (RsaContext, RsaKeyE, TestCtx->RsaE, TestCtx->RsaESize);
  UT_ASSERT_TRUE (Status);

  Status = RsaOaepEncrypt (
             RsaContext,
             (UINT8 *)ClearText,
             ClearTextSize,
             RandSeedIn,
             RandSeedSizeIn,
             DigestLen,
             CipherText,
             CipherTextSize
             );

  return Status;
}

BOOLEAN
EFIAPI
CallRsaOaepDecrypt (
  OAEP_ENC_DEC_TEST_CONTEXT  *TestCtx,
  CONST UINT8                *CipherText,
  UINTN                      CipherTextSize,
  UINT16                     DigestLen,
  UINT8                      **ClearText,
  UINTN                      *ClearTextSize
  )
{
  VOID     *RsaContext = NULL;
  BOOLEAN  Status;

  RsaContext = RsaNew ();
  UT_ASSERT_FALSE (RsaContext == NULL);

  Status = RsaSetKey (RsaContext, RsaKeyN, TestCtx->RsaN, TestCtx->RsaNSize);
  UT_ASSERT_TRUE (Status);

  Status = RsaSetKey (RsaContext, RsaKeyE, TestCtx->RsaE, TestCtx->RsaESize);
  UT_ASSERT_TRUE (Status);

  Status = RsaSetKey (RsaContext, RsaKeyD, TestCtx->RsaD, TestCtx->RsaDSize);
  UT_ASSERT_TRUE (Status);

  Status = RsaOaepDecrypt (
             RsaContext,
             (UINT8 *)CipherText,
             CipherTextSize,
             DigestLen,
             ClearText,
             ClearTextSize
             );

  return Status;
}

UNIT_TEST_STATUS
EFIAPI
TestVerifyEncrypt (
  IN UNIT_TEST_CONTEXT  Context
  )
{
  BOOLEAN                    Status;
  UINT8                      *OutBuffer     = NULL;
  UINTN                      OutBufferSize  = 0;
  UINT8                      *OutBuffer2    = NULL;
  UINTN                      OutBuffer2Size = 0;
  OAEP_ENC_DEC_TEST_CONTEXT  *TestCtx;

  TestCtx = (OAEP_ENC_DEC_TEST_CONTEXT *)Context;

  Status = TestCtx->Encrypt (
                      TestCtx,
                      TestCtx->ClearText,
                      TestCtx->ClearTextSize,
                      TestCtx->PrngSeed,
                      TestCtx->PrngSeedSize,
                      TestCtx->DigestLen,
                      &OutBuffer,
                      &OutBufferSize
                      );
  UT_ASSERT_TRUE (Status);

  Status = TestCtx->Encrypt (
                      TestCtx,
                      TestCtx->ClearText,
                      TestCtx->ClearTextSize,
                      TestCtx->PrngSeed,
                      TestCtx->PrngSeedSize,
                      TestCtx->DigestLen,
                      &OutBuffer2,
                      &OutBuffer2Size
                      );
  UT_ASSERT_TRUE (Status);

  // TRUE - the two OutBuffers are indentical. That means the Oaep encrypt result is incorrect.
  Status = (CompareMem (OutBuffer, OutBuffer2, OutBufferSize >= OutBuffer2Size ? OutBufferSize : OutBuffer2Size) == 0);
  UT_ASSERT_FALSE (Status);

  if (OutBuffer) {
    FreePool (OutBuffer);
    OutBuffer     = NULL;
    OutBufferSize = 0;
  }

  if (OutBuffer2) {
    FreePool (OutBuffer2);
    OutBuffer2     = NULL;
    OutBuffer2Size = 0;
  }

  return UNIT_TEST_PASSED;
}

UNIT_TEST_STATUS
EFIAPI
TestVerifyDecrypt (
  IN UNIT_TEST_CONTEXT  Context
  )
{
  BOOLEAN                    Status;
  UINT8                      *OutBuffer    = NULL;
  UINTN                      OutBufferSize = 0;
  OAEP_ENC_DEC_TEST_CONTEXT  *TestCtx;

  TestCtx = Context;

  Status = TestCtx->Decrypt (
                      TestCtx,
                      TestCtx->CipherText,
                      TestCtx->CipherTextSize,
                      TestCtx->DigestLen,
                      &OutBuffer,
                      &OutBufferSize
                      );
  UT_ASSERT_TRUE (Status);

  UT_ASSERT_TRUE (CompareMem (OutBuffer, TestCtx->ClearText, OutBufferSize >= TestCtx->ClearTextSize ? OutBufferSize : TestCtx->ClearTextSize) == 0);
  UT_ASSERT_TRUE (OutBufferSize == TestCtx->ClearTextSize);

  if (OutBuffer) {
    FreePool (OutBuffer);
    OutBuffer     = NULL;
    OutBufferSize = 0;
  }

  return UNIT_TEST_PASSED;
}

UNIT_TEST_STATUS
EFIAPI
TestVerifyEncryptDecrypt (
  IN UNIT_TEST_CONTEXT  Context
  )
{
  BOOLEAN                    Status;
  UINT8                      *ClearText     = NULL;
  UINTN                      ClearTextSize  = 0;
  UINT8                      *CipherText    = NULL;
  UINTN                      CipherTextSize = 0;
  OAEP_ENC_DEC_TEST_CONTEXT  *TestCtx;

  TestCtx = Context;

  Status = TestCtx->Encrypt (
                      TestCtx,
                      TestCtx->ClearText,
                      TestCtx->ClearTextSize,
                      TestCtx->PrngSeed,
                      TestCtx->PrngSeedSize,
                      TestCtx->DigestLen,
                      &CipherText,
                      &CipherTextSize
                      );
  UT_ASSERT_TRUE (Status);

  Status = TestCtx->Decrypt (
                      TestCtx,
                      CipherText,
                      CipherTextSize,
                      TestCtx->DigestLen,
                      &ClearText,
                      &ClearTextSize
                      );

  if (TestCtx->Expect == UNIT_TEST_PASSED) {
    UT_ASSERT_TRUE (Status);
  } else {
    UT_ASSERT_FALSE (Status);
  }

  if (TestCtx->Expect == UNIT_TEST_PASSED) {
    UT_ASSERT_TRUE (CompareMem (ClearText, TestCtx->ClearText, ClearTextSize >= TestCtx->ClearTextSize ? ClearTextSize : TestCtx->ClearTextSize) == 0);
    UT_ASSERT_TRUE (ClearTextSize == TestCtx->ClearTextSize);
  }

  if (CipherText) {
    FreePool (CipherText);
    CipherText     = NULL;
    CipherTextSize = 0;
  }

  if (ClearText) {
    FreePool (ClearText);
    ClearText     = NULL;
    ClearTextSize = 0;
  }

  return UNIT_TEST_PASSED;
}

UNIT_TEST_STATUS
EFIAPI
TestVerifyPkcs1v2EncryptInterface (
  IN UNIT_TEST_CONTEXT  Context
  )
{
  BOOLEAN  Status;
  UINT8    *OutBuffer;
  UINTN    OutBufferSize;

  Status = Pkcs1v2Encrypt (
             NULL,
             (UINTN)sizeof (SelfTestCert),
             (UINT8 *)Msg1230,
             (UINTN)4,
             NULL,
             0,
             &OutBuffer,
             (UINTN *)&OutBufferSize
             );
  UT_ASSERT_FALSE (Status);

  Status = Pkcs1v2Encrypt (
             SelfTestCert,
             (UINTN)sizeof (SelfTestCert),
             (UINT8 *)Msg1230,
             (UINTN)4,
             NULL,
             0,
             (UINT8 **)NULL,
             (UINTN *)&OutBufferSize
             );
  UT_ASSERT_FALSE (Status);

  Status = Pkcs1v2Encrypt (
             SelfTestCert,
             (UINTN)sizeof (SelfTestCert),
             (UINT8 *)Msg1230,
             (UINTN)4,
             NULL,
             0,
             &OutBuffer,
             (UINTN *)NULL
             );
  UT_ASSERT_FALSE (Status);

  return UNIT_TEST_PASSED;
}

UNIT_TEST_STATUS
EFIAPI
TestVerifyRsaOaepEncryptInterface (
  IN UNIT_TEST_CONTEXT  Context
  )
{
  VOID     *RsaContext = NULL;
  BOOLEAN  Status;
  UINT8    *OutBuffer;
  UINTN    OutBufferSize;

  RsaContext = RsaNew ();
  UT_ASSERT_FALSE (RsaContext == NULL);

  Status = RsaSetKey (RsaContext, RsaKeyN, RsaN, sizeof (RsaN));
  UT_ASSERT_TRUE (Status);

  Status = RsaSetKey (RsaContext, RsaKeyE, RsaE, sizeof (RsaE));
  UT_ASSERT_TRUE (Status);

  Status = RsaOaepEncrypt (
             NULL,
             (UINT8 *)Msg1230,
             (UINTN)4,
             NULL,
             0,
             0,
             &OutBuffer,
             (UINTN *)&OutBufferSize
             );
  UT_ASSERT_FALSE (Status);

  Status = RsaOaepEncrypt (
             RsaContext,
             (UINT8 *)Msg1230,
             (UINTN)4,
             NULL,
             0,
             0,
             (UINT8 **)NULL,
             (UINTN *)&OutBufferSize
             );
  UT_ASSERT_FALSE (Status);

  Status = RsaOaepEncrypt (
             RsaContext,
             (UINT8 *)Msg1230,
             (UINTN)4,
             NULL,
             0,
             0,
             &OutBuffer,
             (UINTN *)NULL
             );
  UT_ASSERT_FALSE (Status);

  return UNIT_TEST_PASSED;
}

OAEP_ENC_DEC_TEST_CONTEXT  mTestVerifyPkcs1v2Msg1230 = {
  .SelfTestCert     = SelfTestCert,
  .SelfTestCertSize = sizeof (SelfTestCert),
  .PrivateKey       = PrivateKey,
  .PrivateKeySize   = sizeof (PrivateKey),
  .RsaN             = NULL,
  .RsaNSize         = 0,
  .RsaE             = NULL,
  .RsaESize         = 0,
  .RsaD             = NULL,
  .RsaDSize         = 0,
  .PrngSeed         = NULL,
  .PrngSeedSize     = 0,
  .ClearText        = Msg1230,
  .ClearTextSize    = sizeof (Msg1230),
  .CipherText       = Ct1230RsaesOaepMdSha1Mgf1Sha1,
  .CipherTextSize   = sizeof (Ct1230RsaesOaepMdSha1Mgf1Sha1),
  .DigestLen        = 0,
  .Encrypt          = CallPkcs1v2Encrypt,
  .Decrypt          = CallPkcs1v2Decrypt,
  .Expect           = UNIT_TEST_PASSED
};

OAEP_ENC_DEC_TEST_CONTEXT  mTestVerifyPkcs1v2Msg1230PrngSeed = {
  .SelfTestCert     = SelfTestCert,
  .SelfTestCertSize = sizeof (SelfTestCert),
  .PrivateKey       = PrivateKey,
  .PrivateKeySize   = sizeof (PrivateKey),
  .RsaN             = NULL,
  .RsaNSize         = 0,
  .RsaE             = NULL,
  .RsaESize         = 0,
  .RsaD             = NULL,
  .RsaDSize         = 0,
  .PrngSeed         = RandSeed,
  .PrngSeedSize     = sizeof (RandSeed),
  .ClearText        = Msg1230,
  .ClearTextSize    = sizeof (Msg1230),
  .CipherText       = Ct1230RsaesOaepMdSha1Mgf1Sha1,
  .CipherTextSize   = sizeof (Ct1230RsaesOaepMdSha1Mgf1Sha1),
  .DigestLen        = 0,
  .Encrypt          = CallPkcs1v2Encrypt,
  .Decrypt          = CallPkcs1v2Decrypt,
  .Expect           = UNIT_TEST_PASSED
};

OAEP_ENC_DEC_TEST_CONTEXT  mTestVerifyRsaOaepMsg1230 = {
  .SelfTestCert     = NULL,
  .SelfTestCertSize = 0,
  .PrivateKey       = NULL,
  .PrivateKeySize   = 0,
  .RsaN             = RsaN,
  .RsaNSize         = sizeof (RsaN),
  .RsaE             = RsaE,
  .RsaESize         = sizeof (RsaE),
  .RsaD             = RsaD,
  .RsaDSize         = sizeof (RsaD),
  .PrngSeed         = NULL,
  .PrngSeedSize     = 0,
  .ClearText        = Msg1230,
  .ClearTextSize    = sizeof (Msg1230),
  .CipherText       = Ct1230RsaesOaepMdSha1Mgf1Sha1,
  .CipherTextSize   = sizeof (Ct1230RsaesOaepMdSha1Mgf1Sha1),
  .DigestLen        = 0,
  .Encrypt          = CallRsaOaepEncrypt,
  .Decrypt          = CallRsaOaepDecrypt,
  .Expect           = UNIT_TEST_PASSED
};

OAEP_ENC_DEC_TEST_CONTEXT  mTestVerifyRsaOaepMsg1230PrngSeed = {
  .SelfTestCert     = NULL,
  .SelfTestCertSize = 0,
  .PrivateKey       = NULL,
  .PrivateKeySize   = 0,
  .RsaN             = RsaN,
  .RsaNSize         = sizeof (RsaN),
  .RsaE             = RsaE,
  .RsaESize         = sizeof (RsaE),
  .RsaD             = RsaD,
  .RsaDSize         = sizeof (RsaD),
  .PrngSeed         = RandSeed,
  .PrngSeedSize     = sizeof (RandSeed),
  .ClearText        = Msg1230,
  .ClearTextSize    = sizeof (Msg1230),
  .CipherText       = Ct1230RsaesOaepMdSha1Mgf1Sha1,
  .CipherTextSize   = sizeof (Ct1230RsaesOaepMdSha1Mgf1Sha1),
  .DigestLen        = 0,
  .Encrypt          = CallRsaOaepEncrypt,
  .Decrypt          = CallRsaOaepDecrypt,
  .Expect           = UNIT_TEST_PASSED
};

OAEP_ENC_DEC_TEST_CONTEXT  mTestVerifyPkcs1v2EncryptRsaOaepDecrypt = {
  .SelfTestCert     = SelfTestCert,
  .SelfTestCertSize = sizeof (SelfTestCert),
  .PrivateKey       = PrivateKey,
  .PrivateKeySize   = sizeof (PrivateKey),
  .RsaN             = RsaN,
  .RsaNSize         = sizeof (RsaN),
  .RsaE             = RsaE,
  .RsaESize         = sizeof (RsaE),
  .RsaD             = RsaD,
  .RsaDSize         = sizeof (RsaD),
  .ClearText        = Msg1230,
  .ClearTextSize    = sizeof (Msg1230),
  .CipherText       = Ct1230RsaesOaepMdSha1Mgf1Sha1,
  .CipherTextSize   = sizeof (Ct1230RsaesOaepMdSha1Mgf1Sha1),
  .DigestLen        = 0,
  .Encrypt          = CallPkcs1v2Encrypt,
  .Decrypt          = CallRsaOaepDecrypt,
  .Expect           = UNIT_TEST_PASSED
};

OAEP_ENC_DEC_TEST_CONTEXT  mTestVerifyRsaOaepEncryptPkcs1v2Decrypt = {
  .SelfTestCert     = SelfTestCert,
  .SelfTestCertSize = sizeof (SelfTestCert),
  .PrivateKey       = PrivateKey,
  .PrivateKeySize   = sizeof (PrivateKey),
  .RsaN             = RsaN,
  .RsaNSize         = sizeof (RsaN),
  .RsaE             = RsaE,
  .RsaESize         = sizeof (RsaE),
  .RsaD             = RsaD,
  .RsaDSize         = sizeof (RsaD),
  .ClearText        = Msg1230,
  .ClearTextSize    = sizeof (Msg1230),
  .CipherText       = Ct1230RsaesOaepMdSha1Mgf1Sha1,
  .CipherTextSize   = sizeof (Ct1230RsaesOaepMdSha1Mgf1Sha1),
  .DigestLen        = 0,
  .Encrypt          = CallRsaOaepEncrypt,
  .Decrypt          = CallPkcs1v2Decrypt,
  .Expect           = UNIT_TEST_PASSED
};

OAEP_ENC_DEC_TEST_CONTEXT  mTestVerifyRsaesOaep2048MdDefaultBgf1Default = {
  .SelfTestCert     = NULL,
  .SelfTestCertSize = 0,
  .PrivateKey       = NULL,
  .PrivateKeySize   = 0,
  .RsaN             = RsaN,
  .RsaNSize         = sizeof (RsaN),
  .RsaE             = RsaE,
  .RsaESize         = sizeof (RsaE),
  .RsaD             = RsaD,
  .RsaDSize         = sizeof (RsaD),
  .ClearText        = Msg1230,
  .ClearTextSize    = sizeof (Msg1230),
  .CipherText       = Ct1230RsaesOaepMdSha1Mgf1Sha1,
  .CipherTextSize   = sizeof (Ct1230RsaesOaepMdSha1Mgf1Sha1),
  .DigestLen        = 0,
  .Encrypt          = CallRsaOaepEncrypt,
  .Decrypt          = CallRsaOaepDecrypt,
  .Expect           = UNIT_TEST_PASSED
};

OAEP_ENC_DEC_TEST_CONTEXT  mTestVerifyRsaesOaep2048MdSha1Mgf1Sha1 = {
  .SelfTestCert     = NULL,
  .SelfTestCertSize = 0,
  .PrivateKey       = NULL,
  .PrivateKeySize   = 0,
  .RsaN             = RsaN,
  .RsaNSize         = sizeof (RsaN),
  .RsaE             = RsaE,
  .RsaESize         = sizeof (RsaE),
  .RsaD             = RsaD,
  .RsaDSize         = sizeof (RsaD),
  .ClearText        = Msg1230,
  .ClearTextSize    = sizeof (Msg1230),
  .CipherText       = Ct1230RsaesOaepMdSha1Mgf1Sha1,
  .CipherTextSize   = sizeof (Ct1230RsaesOaepMdSha1Mgf1Sha1),
  .DigestLen        = SHA1_DIGEST_SIZE,
  .Encrypt          = CallRsaOaepEncrypt,
  .Decrypt          = CallRsaOaepDecrypt,
  .Expect           = UNIT_TEST_PASSED
};

OAEP_ENC_DEC_TEST_CONTEXT  mTestVerifyRsaesOaep2048MdSha256Mgf1Sha256 = {
  .SelfTestCert     = NULL,
  .SelfTestCertSize = 0,
  .PrivateKey       = NULL,
  .PrivateKeySize   = 0,
  .RsaN             = RsaN,
  .RsaNSize         = sizeof (RsaN),
  .RsaE             = RsaE,
  .RsaESize         = sizeof (RsaE),
  .RsaD             = RsaD,
  .RsaDSize         = sizeof (RsaD),
  .ClearText        = Msg1230,
  .ClearTextSize    = sizeof (Msg1230),
  .CipherText       = Ct1230RsaesOaep2048MdSha256Mgf1Sha256,
  .CipherTextSize   = sizeof (Ct1230RsaesOaep2048MdSha256Mgf1Sha256),
  .DigestLen        = SHA256_DIGEST_SIZE,
  .Encrypt          = CallRsaOaepEncrypt,
  .Decrypt          = CallRsaOaepDecrypt,
  .Expect           = UNIT_TEST_PASSED
};

TEST_DESC  mOaepTest[] = {
  //
  // -----Description--------------------------------------Class----------------------Function-----------------Pre---Post--Context
  //
  // Pkcs1v2Encrypt / Decrypt
  { "Pkcs1v2Encrypt (Interface)",                   "CryptoPkg.BaseCryptLib.Pkcs1v2Encrypt.Interface",                   TestVerifyPkcs1v2EncryptInterface, NULL, NULL, &mTestVerifyPkcs1v2Msg1230                    },
  { "Pkcs1v2Encrypt (NoSeed)",                      "CryptoPkg.BaseCryptLib.Pkcs1v2Encrypt.NoSeed",                      TestVerifyEncrypt,                 NULL, NULL, &mTestVerifyPkcs1v2Msg1230                    },
  { "Pkcs1v2Encrypt (Seeded)",                      "CryptoPkg.BaseCryptLib.Pkcs1v2Encrypt.Seeded",                      TestVerifyEncrypt,                 NULL, NULL, &mTestVerifyPkcs1v2Msg1230PrngSeed            },
  { "Pkcs1v2Decrypt",                               "CryptoPkg.BaseCryptLib.Pkcs1v2Decrypt",                             TestVerifyDecrypt,                 NULL, NULL, &mTestVerifyPkcs1v2Msg1230                    },
  { "Pkcs1v2EncryptDecrypt",                        "CryptoPkg.BaseCryptLib.Pkcs1v2EncryptDecrypt",                      TestVerifyEncryptDecrypt,          NULL, NULL, &mTestVerifyPkcs1v2Msg1230                    },

  // RsaOaepEncrypt / Decrypt
  { "RsaOaepEncrypt (Interface)",                   "CryptoPkg.BaseCryptLib.RsaOaepEncrypt.Interface",                   TestVerifyRsaOaepEncryptInterface, NULL, NULL, &mTestVerifyRsaOaepMsg1230                    },
  { "RsaOaepEncrypt (NoSeed)",                      "CryptoPkg.BaseCryptLib.RsaOaepEncrypt.NoSeed",                      TestVerifyEncrypt,                 NULL, NULL, &mTestVerifyRsaOaepMsg1230                    },
  { "RsaOaepEncrypt (Seeded)",                      "CryptoPkg.BaseCryptLib.RsaOaepEncrypt.Seeded",                      TestVerifyEncrypt,                 NULL, NULL, &mTestVerifyRsaOaepMsg1230PrngSeed            },
  { "RsaOaepDecrypt",                               "CryptoPkg.BaseCryptLib.RsaOaepDecrypt",                             TestVerifyDecrypt,                 NULL, NULL, &mTestVerifyRsaOaepMsg1230                    },
  { "RsaOaepEncryptDecrypt",                        "CryptoPkg.BaseCryptLib.RsaOaepEncryptDecrypt",                      TestVerifyEncryptDecrypt,          NULL, NULL, &mTestVerifyRsaOaepMsg1230                    },

  // Mix interfaces
  { "RsaOaepEncryptPkcs1v2Decrypt",                 "CryptoPkg.BaseCryptLib.RsaOaepEncryptPkcs1v2Decrypt",               TestVerifyEncryptDecrypt,          NULL, NULL, &mTestVerifyRsaOaepEncryptPkcs1v2Decrypt      },
  { "Pkcs1v2EncryptRsaOaepDecrypt",                 "CryptoPkg.BaseCryptLib.Pkcs1v2EncryptRsaOaepDecrypt",               TestVerifyEncryptDecrypt,          NULL, NULL, &mTestVerifyPkcs1v2EncryptRsaOaepDecrypt      },

  // Message digest default / MGF1 default (SHA1)
  { "RsaOaepEncrypt (MdDefaultMgf1Default)",        "CryptoPkg.BaseCryptLib.RsaOaepEncrypt.MdDefaultMgf1Default",        TestVerifyEncrypt,                 NULL, NULL, &mTestVerifyRsaesOaep2048MdDefaultBgf1Default },
  { "RsaOaepDecrypt (MdDefaultMgf1Default)",        "CryptoPkg.BaseCryptLib.RsaOaepDecrypt.MdDefaultMgf1Default",        TestVerifyDecrypt,                 NULL, NULL, &mTestVerifyRsaesOaep2048MdDefaultBgf1Default },
  { "RsaOaepEncryptDecrypt (MdDefaultMgf1Default)", "CryptoPkg.BaseCryptLib.RsaOaepEncryptDecrypt.MdDefaultMgf1Default", TestVerifyEncryptDecrypt,          NULL, NULL, &mTestVerifyRsaesOaep2048MdDefaultBgf1Default },

  // Message digest SHA1 / MGF1 SHA1
  { "RsaOaepEncrypt (MdSha1Bgf1Sha1",               "CryptoPkg.BaseCryptLib.RsaOaepEncrypt.MdSha1Bgf1Sha1",              TestVerifyEncrypt,                 NULL, NULL, &mTestVerifyRsaesOaep2048MdSha1Mgf1Sha1       },
  { "RsaOaepDecrypt (MdSha1Bgf1Sha1)",              "CryptoPkg.BaseCryptLib.RsaOaepDecrypt.MdSha1Bgf1Sha1",              TestVerifyDecrypt,                 NULL, NULL, &mTestVerifyRsaesOaep2048MdSha1Mgf1Sha1       },
  { "RsaOaepEncryptDecrypt (MdSha1Bgf1Sha1)",       "CryptoPkg.BaseCryptLib.RsaOaepEncryptDecrypt.MdSha1Bgf1Sha1",       TestVerifyEncryptDecrypt,          NULL, NULL, &mTestVerifyRsaesOaep2048MdSha1Mgf1Sha1       },

  // Message digest SHA256 / MGF1 SHA256
  { "RsaOaepEncrypt (MdSha256Bgf1Sha256)",          "CryptoPkg.BaseCryptLib.RsaOaepEncrypt.MdSha256Bgf1Sha256",          TestVerifyEncrypt,                 NULL, NULL, &mTestVerifyRsaesOaep2048MdSha256Mgf1Sha256   },
  { "RsaOaepDecrypt (MdSha256Bgf1Sha256)",          "CryptoPkg.BaseCryptLib.RsaOaepDecrypt.MdSha256Bgf1Sha256",          TestVerifyDecrypt,                 NULL, NULL, &mTestVerifyRsaesOaep2048MdSha256Mgf1Sha256   },
  { "RsaOaepEncryptDecrypt (MdSha256Bgf1Sha256)",   "CryptoPkg.BaseCryptLib.RsaOaepEncryptDecryptMdSha256Bgf1Sha256",    TestVerifyEncryptDecrypt,          NULL, NULL, &mTestVerifyRsaesOaep2048MdSha256Mgf1Sha256   },
};

UINTN  mOaepTestNum = ARRAY_SIZE (mOaepTest);
