# Webmention Server with Vouch Support

This example demonstrates a complete Webmention server implementation that includes Vouch anti-spam validation. The server accepts Webmentions and validates them according to the Vouch protocol specification.

## Overview

The server implements the following Vouch protocol features:

- **Authority-based approval**: Checks if the sender's domain is in an approved list
- **Vouch validation**: Accepts vouched webmentions from unapproved domains
- **HTTP status codes**: Returns appropriate status codes per the Vouch spec
- **Web interface**: Displays all received webmentions with validation status

## Setup

1. Ensure Rust is installed (see rust-toolchain.toml for version).
2. Navigate to this directory.
3. Run `cargo build` to compile.

## Running

Execute `cargo run` to start the server. Visit `http://localhost:8080` in your browser to view received webmentions.

## How It Works

### Domain Approval

This example uses the `AlwaysAllow` implementation, which approves all domains. In a real implementation, you would:

1. Fetch h-cards from `https://indieweb.org/chat-names`
2. Extract authorities from IndieWeb community members
3. Cache the list and check senders against it

### Vouch Validation

When a webmention is received:

1. **Extract sender authority** from the source URL
2. **Check approval**: If approved, accept the webmention
3. **Check vouch**: If not approved but vouch provided, validate the vouch
4. **Return status**: 200 (accepted), 400 (invalid), or 449 (retry with vouch)

### HTTP Status Codes

- `200 OK`: Webmention accepted
- `400 Bad Request`: Invalid URLs, unapproved vouch, or validation failure
- `449 Retry With`: Authority not approved and no vouch provided

## Testing

Use the web interface at `http://localhost:8080` to see all received webmentions, or use curl commands:

### Send from Approved Authority
```bash
curl -X POST http://localhost:8080/webmention \
  -H "Content-Type: application/x-www-form-urlencoded" \
  -d "source=https://example.com/post/1&target=http://localhost:8080/post/1"
```

### Send from Unapproved Authority (No Vouch)
```bash
curl -X POST http://localhost:8080/webmention \
  -H "Content-Type: application/x-www-form-urlencoded" \
  -d "source=https://unknown-site.com/post/1&target=http://localhost:8080/post/1"
```

### Send from Unapproved Authority (With Vouch)
```bash
curl -X POST http://localhost:8080/webmention \
  -H "Content-Type: application/x-www-form-urlencoded" \
  -d "source=https://unknown-site.com/post/1&target=http://localhost:8080/post/1&vouch=https://approved-site.com/page"
```

## Implementation Details

### Architecture

- **Axum web framework** for HTTP handling
- **Minijinja templates** for HTML rendering
- **Pico CSS** for styling
- **Shared state** for webmention storage
- **Async processing** for all operations

### Key Components

- `main.rs`: Server setup and routing
- `webmention.rs`: Webmention endpoint handling and validation
- `templates/`: HTML templates for the web interface
- `SharedState`: Domain approver configuration
- `WebmentionState`: Storage for received webmentions

### Vouch Protocol Compliance

This implementation follows the [Vouch specification](https://indieweb.org/Vouch) including:

- Authority extraction from URLs
- Proper HTTP status code responses
- Vouch URL validation (placeholder in example)
- Backward compatibility with standard Webmention

## Extending the Example

To make this a production-ready server:

1. **Replace AlwaysAllow** with a real domain approval implementation
2. **Add vouch validation** that actually checks if vouch URLs link to sender domains
3. **Add persistence** to store webmentions across server restarts
4. **Add authentication** for managing approved domains
5. **Add rate limiting** to prevent abuse

## Related Links

- [Vouch Protocol Specification](https://indieweb.org/Vouch)
- [Webmention Specification](https://www.w3.org/TR/webmention/)
- [IndieWeb Chat Names](https://indieweb.org/chat-names)