/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright 2009--2026 by Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// StdLib includes


/////////////////////// Qt includes
#include <QDebug>
#include <QObject>
#include <QDialog>
#include <QAction>
#include <QSpinBox>
#include <QSettings>
#include <QMainWindow>
#include <QSvgWidget>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QFormLayout>
#include <QtCore/QVariant>
#include <QtWidgets/QScrollArea>
#include <QThread>
#include <QFileDialog>
#include <QMessageBox>


/////////////////////// pappsomspp includes


/////////////////////// libXpertMassCore includes
#include "MsXpS/libXpertMassCore/Utils.hpp"


/////////////////////// libXpertMassGui includes
#include <MsXpS/libXpertMassGui/ActionManagerTableWidget.hpp>
#include <MsXpS/libXpertMassGui/ToleranceWidget.hpp>

/////////////////////// Local includes
#include "ApplicationPreferencesWnd.hpp"
#include "Application.hpp"
#include "ProgramWindow.hpp"

#include "ui_ApplicationPreferencesWnd.h"
#include "ui_ApplicationPreferencesWndGeneralPageWidget.h"
#include "ui_ApplicationPreferencesWndDiscoveriesPageWidget.h"
#include "ui_DecimalPlacesPreferencesWidget.h"
#include "ui_PeakPickingPreferencesWidget.h"
#include "ui_MaxThreadCountPreferencesWidget.h"
#include "ui_LowMassDeconvolverPreferencesWidget.h"

namespace MsXpS
{
namespace MineXpert
{

ApplicationPreferencesWnd::ApplicationPreferencesWnd(
  const QString &application_name,
  const QString &description,
  QWidget *parent_p)
  : QMainWindow(parent_p),
    mp_ui(new Ui::ApplicationPreferencesWnd),
    mp_generalPage_ui(new Ui::ApplicationPreferencesWndGeneralPageWidget),
    mp_discoveriesPage_ui(
      new Ui::ApplicationPreferencesWndDiscoveriesPageWidget),
    mp_decimalPlacesPreferencesWidget_ui(
      new Ui::DecimalPlacesPreferencesWidget),
    mp_maxThreadCountPreferencesWidget_ui(
      new Ui::MaxThreadCountPreferencesWidget),
    mp_peakPickingPreferencesWidget_ui(new Ui::PeakPickingPreferencesWidget),
    mp_lowMassDeconvolverPreferencesWidget_ui(
      new Ui::LowMassDeconvolverPreferencesWidget),
    m_applicationName(application_name),
    m_windowDescription(description)
{
  if(parent_p == nullptr)
    qFatal() << "Programming error. Pointer cannot be nullptr.";

  mp_ui->setupUi(this);
  // The general page widget is preformed and we'll need to set
  // the corresponding widget as the first section.
  mp_generalPage_ui->setupUi(this);
  // The discoveries page widget is preformed.
  mp_discoveriesPage_ui->setupUi(this);
  // The decimal places widget that we use the group box from
  // to stuff in the generalPage.
  mp_decimalPlacesPreferencesWidget_ui->setupUi(this);
  // The maximum thread count widget -> generalPage.
  mp_maxThreadCountPreferencesWidget_ui->setupUi(this);
  // The peak picking preferences -> generalPage.
  mp_peakPickingPreferencesWidget_ui->setupUi(this);
  // The low mass deconvolver preferences -> generalPage.
  mp_lowMassDeconvolverPreferencesWidget_ui->setupUi(this);

  mp_ui->mainTitleLabel->setText(
    QString("Preferences for %1").arg(m_applicationName));

  connect(mp_ui->sectionsListWidget,
          &QListWidget::currentRowChanged,
          mp_ui->sectionsStackedWidget,
          &QStackedWidget::setCurrentIndex);

  connect(mp_ui->applyPushButton,
          &QPushButton::clicked,
          this,
          &ApplicationPreferencesWnd::apply);
  connect(mp_ui->closePushButton,
          &QPushButton::clicked,
          this,
          &ApplicationPreferencesWnd::hide);

  // Update the window title because the window title element in mp_ui->might be
  // either erroneous or empty.
  setWindowTitle(
    QString("%1 - %2").arg(m_applicationName, m_windowDescription));

  mp_ui->sectionsListWidget->setViewMode(QListView::IconMode);
  mp_ui->sectionsListWidget->setFlow(
    QListView::TopToBottom);                     // keeps a vertical layout
  mp_ui->sectionsListWidget->setWrapping(false); // one item per row
  mp_ui->sectionsListWidget->setResizeMode(QListView::Adjust);
  mp_ui->sectionsListWidget->setMaximumWidth(85);
  mp_ui->sectionsListWidget->setIconSize(QSize(80, 80)); // adjust as needed
  mp_ui->sectionsListWidget->setGridSize(
    QSize(80, 100)); // must be large enough for icon + text
  mp_ui->sectionsListWidget->setWordWrap(true);
  mp_ui->sectionsListWidget->setTextElideMode(Qt::ElideNone);

  readSettings();

  setupWindow();
}

ApplicationPreferencesWnd::~ApplicationPreferencesWnd()
{
  // qDebug() << "Is visible:" << isVisible();
  writeSettings();
}

/*!
\brief Saves the settings of this dialog window to the application
configuration file (\a config_settings_file_path).

The saved configuration is read from the file to set back the dialog window in
the same status.
*/
void
ApplicationPreferencesWnd::writeSettings() const
{
  QString settings_file_path =
    static_cast<Application *>(QCoreApplication::instance())
      ->getUserConfigSettingsFilePath();

  QSettings settings(settings_file_path, QSettings::IniFormat);

  settings.beginGroup("ApplicationPreferencesWnd");

  settings.setValue("geometry", saveGeometry());

  // Sanity check. Works.
  // int current_stacked_widget_page_index =
  //   mp_ui->sectionsStackedWidget->currentIndex();
  // qDebug() << "Current page index:" << current_stacked_widget_page_index;

  int current_list_widget_page_index =
    mp_ui->sectionsListWidget->currentIndex().row();
  // qDebug() << "Current page index:" << current_list_widget_page_index;

  settings.setValue("currentPageIndex", current_list_widget_page_index);
  settings.setValue("isVisible", isVisible());

  settings.endGroup();
  settings.sync();

  qDebug() << "Done writing the application preferences wnd settings to"
           << settings_file_path;
}

/*!
\brief Reads the settings of this dialog window from the application
configuration file (\a config_settings_file_path).

The configuration is read from the file to set back the dialog window in
the same status.
*/
void
ApplicationPreferencesWnd::readSettings()
{
  QSettings settings(libXpertMassCore::Utils::configSettingsFilePath,
                     QSettings::IniFormat);

  settings.beginGroup("ApplicationPreferencesWnd");

  // We'll set the stored current page index later.

  setVisible(settings.value("isVisible").toBool());

  restoreGeometry(settings.value("geometry").toByteArray());

  settings.endGroup();
}

void
ApplicationPreferencesWnd::setupWindow()
{
  setupGeneralPage();
  setupShortcutsPage();
  setupDiscoveriesPage();

  QSettings settings(libXpertMassCore::Utils::configSettingsFilePath,
                     QSettings::IniFormat);

  settings.beginGroup("ApplicationPreferencesWnd");

  bool ok;
  int current_page_index = settings.value("currentPageIndex", 0).toInt(&ok);
  // qDebug() << "Current page index:" << current_page_index;
  if(!ok)
    qFatal() << "Programming error.";

  mp_ui->sectionsListWidget->setCurrentRow(current_page_index);

  settings.endGroup();
}

void
ApplicationPreferencesWnd::setupGeneralPage()
{
  // The page is available as a widget (see *this constructor)
  // as mp_generalPage_ui->scrollArea in which we need to pack
  // all the widgets we need in order. Each configuration bit
  // needs to be in a QGroupBox.

  // The general page has widgets that can be configured here.
  setupMaxThreadCountWidget();
  setupDecimalPlacesPreferencesWidget();
  setupPeakPickingPreferencesWidget();
  setupLowMassDeconvolverPreferencesWidget();

  // We want to push all the group boxes to the top.

  insertSectionListItem("General",
                        QIcon(":images/icons/svg/MineXpert3.svg"),
                        mp_generalPage_ui->scrollArea,
                        Pages::GENERAL);
}

void
ApplicationPreferencesWnd::setupMaxThreadCountWidget()
{
  // This is the main page scroll area widget contents layout.
  // We will end up packing into this layout (towards the bottom) each newly
  // created QGroupBox widget.
  QVBoxLayout *layout_p = mp_generalPage_ui->mainWidgetContentsLayout;

  // The group box widget that will contain all we need to provide the user
  // with for them to configure the max count of threads.
  QGroupBox *group_box_p =
    mp_maxThreadCountPreferencesWidget_ui->maxThreadCountPreferencesGroupBox;

  // We need to setup the icon in the corresponding frame.
  QFrame *icon_frame_p = mp_maxThreadCountPreferencesWidget_ui->iconFrame;
  icon_frame_p->setFrameShape(QFrame::StyledPanel);
  icon_frame_p->setFrameShadow(QFrame::Raised);
  QHBoxLayout *icon_frame_layout_p = new QHBoxLayout(icon_frame_p);
  icon_frame_layout_p->setContentsMargins(4, 4, 4, 4);
  auto *svg =
    new QSvgWidget(":/images/svg/thread-count-slider.svg", icon_frame_p);
  svg->setFixedSize(48, 48); // choose size explicitly
  icon_frame_layout_p->addWidget(svg);

  std::size_t ideal_thread_count = QThread::idealThreadCount();
  QSpinBox *max_thread_count_spin_box_p =
    mp_maxThreadCountPreferencesWidget_ui->maxThreadCountSpinBox;
  max_thread_count_spin_box_p->setRange(1, ideal_thread_count);
  max_thread_count_spin_box_p->setValue(ideal_thread_count);

  connect(max_thread_count_spin_box_p,
          &QSpinBox::valueChanged,
          this,
          [&](const int &value) {
            static_cast<Application *>(qApp)
              ->getProgramWindow()
              ->setMaxThreadUseCount(value);
          });

  layout_p->addWidget(group_box_p);
}

void
ApplicationPreferencesWnd::setupDecimalPlacesPreferencesWidget()
{
  // This is the main page scroll area widget contents layout.
  // We will end up packing into this layout (towards the bottom) each newly
  // created QGroupBox widget.
  QVBoxLayout *layout_p = mp_generalPage_ui->mainWidgetContentsLayout;

  // The group box widget that will contain all we need to provide the user
  // with for them to configure the max count of threads.
  QGroupBox *group_box_p =
    mp_decimalPlacesPreferencesWidget_ui->decimalPlacesPreferencesGroupBox;

  layout_p->addWidget(group_box_p);
}

void
ApplicationPreferencesWnd::setupPeakPickingPreferencesWidget()
{
  // This is the main page scroll area widget contents layout.
  // We will end up packing into this layout (towards the bottom) each newly
  // created QGroupBox widget.
  QVBoxLayout *layout_p = mp_generalPage_ui->mainWidgetContentsLayout;

  // The group box widget that will contain all we need to provide the user
  // with for them to configure the max count of threads.
  QGroupBox *group_box_p =
    mp_peakPickingPreferencesWidget_ui->peakPickingPreferencesGroupBox;

  // Read the peak picking preferences and set the values to the widgets.
  readPeakPickingPreferences();

  layout_p->addWidget(group_box_p);

  // qDebug() << "Done.";
}

void
ApplicationPreferencesWnd::setupLowMassDeconvolverPreferencesWidget()
{
  // This is the main page scroll area widget contents layout.
  // We will end up packing into this layout (towards the bottom) each newly
  // created QGroupBox widget.
  QVBoxLayout *layout_p = mp_generalPage_ui->mainWidgetContentsLayout;

  // The group box widget that will contain all we need to provide the user
  // with for them to configure the max count of threads.
  QGroupBox *group_box_p = mp_lowMassDeconvolverPreferencesWidget_ui
                             ->lowMassDeconvolverPreferencesGroupBox;

  connect(mp_lowMassDeconvolverPreferencesWidget_ui->massToleranceDoubleSpinBox,
          &QDoubleSpinBox::valueChanged,
          this,
          [&](double new_value) {
            qDebug() << "PPM Tolerance changed:" << new_value;
          });

  connect(
    mp_lowMassDeconvolverPreferencesWidget_ui->selectIsotopicDataFilePushButton,
    &QPushButton::clicked,
    this,
    [&]() {
      QString file_name =
        QFileDialog::getOpenFileName(this, "Open isotopic data file");

      if(!file_name.isEmpty())
        {
          mp_lowMassDeconvolverPreferencesWidget_ui->isotopicDataFileLineEdit
            ->setText(file_name);
        }
    });

  connect(mp_lowMassDeconvolverPreferencesWidget_ui->historyItemTitleComboBox,
          &QComboBox::currentTextChanged,
          this,
          [&](const QString &current_text) {
            populateLowMassDeconvolverPreferencesForHistoryItem(current_text);
          });

  connect(mp_lowMassDeconvolverPreferencesWidget_ui->addToHistoryPushButton,
          &QPushButton::clicked,
          this,
          [&]() {
            addLowMassDeconvolverPreferencesToHistory();
          });

  connect(
    mp_lowMassDeconvolverPreferencesWidget_ui->removeFromHistoryPushButton,
    &QPushButton::clicked,
    this,
    [&]() {
      removeLowMassDeconvolverPreferencesFromHistory();
    });

  connect(mp_lowMassDeconvolverPreferencesWidget_ui->saveHistoryItem,
          &QPushButton::clicked,
          this,
          [&]() {
            updateLowMassDeconvolverPreferencesCurrentHistoryItem();
          });

  // Read the peak picking preferences and set the values to the widgets.
  readLowMassDeconvolverPreferences();

  layout_p->addWidget(group_box_p);

  // qDebug() << "Done.";
}

void
ApplicationPreferencesWnd::setupDiscoveriesPage()
{
  // The widget we want to add as a new preferences section is
  QIcon svg_icon(":/images/svg/discoveries-preferences.svg");
  insertSectionListItem("Discoveries",
                        svg_icon,
                        mp_discoveriesPage_ui->scrollArea,
                        Pages ::DISCOVERIES);

  QSharedPointer<DiscoveriesPreferences> discoveries_preferences_qsp =
    QSharedPointer<DiscoveriesPreferences>::create();

  // qDebug() << "The discoveries_preferences_qsp:"
  //          << discoveries_preferences_qsp.get();

  QSharedPointer<QList<QPlainTextEdit *>> plain_text_edit_list_qsp =
    QSharedPointer<QList<QPlainTextEdit *>>::create();

  mp_discoveriesPage_ui->scrollArea->setProperty(
    "discoveries_preferences_qsp",
    QVariant::fromValue(discoveries_preferences_qsp));
  mp_discoveriesPage_ui->scrollArea->setProperty(
    "plain_text_edit_list_qsp", QVariant::fromValue(plain_text_edit_list_qsp));

  // Create the format tabs, using the data in the
  // m_dataFormatStringSpecifHash member of analysis_preferences:

  for(int iter = 0; iter < FormatType::LAST; ++iter)
    {
      DataFormatStringSpecif *specif =
        discoveries_preferences_qsp->m_dataFormatStringSpecifHash.value(iter);

      if(specif == nullptr)
        {
          qFatal() << "Programming error.";
          return;
        }

      // First allocate an new plain text edit widget that we'll set to the
      // tab when we create it below. We want to store a pointer to the text
      // edit widget for later use.
      QPlainTextEdit *newEdit =
        new QPlainTextEdit(mp_discoveriesPage_ui->formatsTabWidget);
      plain_text_edit_list_qsp->append(newEdit);

      mp_discoveriesPage_ui->formatsTabWidget->insertTab(
        iter, newEdit, specif->m_label);
    }

  // Read the settings, if any.
  QSettings settings(static_cast<Application *>(QCoreApplication::instance())
                       ->getUserConfigSettingsFilePath(),
                     QSettings::IniFormat);

  settings.beginGroup("DiscoveriesPreferences");

  QStringList historyList = settings.value("history").toStringList();
  mp_discoveriesPage_ui->historyComboBox->insertItems(0, historyList);

  mp_discoveriesPage_ui->destSplitter->restoreState(
    settings.value("destSplitter").toByteArray());
  mp_discoveriesPage_ui->formatSplitter->restoreState(
    settings.value("formatSplitter").toByteArray());

  // Also restore the geometry of the window.
  restoreGeometry(settings.value("geometry").toByteArray());

  // Now the various format strings for the tic, ms, dt tab widgets, if any.

  for(int iter = 0; iter < FormatType::LAST; ++iter)
    {
      const DataFormatStringSpecif *specif_p =
        discoveries_preferences_qsp->m_dataFormatStringSpecifHash.value(iter);

      if(specif_p == nullptr)
        qFatal() << "Programming error.";

      QString tabString =
        mp_discoveriesPage_ui->formatsTabWidget->tabText(iter);
      QString settingsKey = QString("%1_formatString").arg(tabString);

      QString settingsValue = settings.value(settingsKey).toString();

      QPlainTextEdit *textEdit = static_cast<QPlainTextEdit *>(
        mp_discoveriesPage_ui->formatsTabWidget->widget(iter));

      textEdit->setPlainText(settingsValue);

      // qDebug() << "Reading setting key/value for iter"
      //<< iter << settingsKey << settingsValue;
    }

  settings.endGroup();


  // At this point we need to display the help text that shows the various
  // possibilities for the format string.

  QString helpText;

  helpText +=
    "The formatting of the text describing the exploration session "
    "discovery is described below. It uses specific format patterns, like '%f' "
    "that will be replaced with corresponding value, like the filename, for "
    "example.\n\n";

  helpText +=
    "The numeric values that can be used to describe a discovery might "
    "be of differnt types: "
    "  - floats with decimals (m/z value)"
    "  - integers without decimals (charge z)\n\n";

  helpText +=
    "When float numeric values require a specific number of decimals, "
    "that can be specified using the decimals specifier of the form '.3', for "
    "example "
    "if the float number needs to be printed out with three decimas.\n\n";

  helpText +=
    "Float values for which no decimal specifier is set are printed "
    "out with 2 decimals by default.\n\n";

  helpText +=
    "If a float value is requied without decimal, use the '.0' decimal "
    "specifier.\n\n";

  helpText += "These are the available format strings:\n\n";

  helpText += "%f: mass spectrometry data file name\n";
  helpText += "%s: sample name\n";

  helpText += "%X (or %X.3): value on the X axis (no unit)\n";
  helpText += "%Y (or %Y.1): value on the Y axis at cursor point(no unit)\n";
  helpText += "%C (or %C.1): value on the Y axis on the curve (no unit)\n";

  helpText +=
    "%x (or %x.6): delta value on the X axis (when appropriate, no unit)\n";
  helpText +=
    "%y (or %y.1): delta value on the Y axis (when appropriate, no unit)\n";
  helpText +=
    "%c (or %c.1): delta value on the Y axis on the curve (no "
    "unit)\n";

  helpText +=
    "%I (or %I.1): TIC intensity after TIC integration over a graph range\n";

  helpText += "%b (or %b.3): X axis range begin value (where applicable)\n";
  helpText += "%e (or %e.3): X axis range end value (where applicable)\n";

  helpText += "\nMass spec plot widgets:\n";
  helpText += "\t%z : charge\n";
  helpText +=
    "\t%M (or %M.5 for small mass ions; %M.2 for medium-mass ions): Mr\n";

  mp_discoveriesPage_ui->formatDescriptionPlainTextEdit->appendPlainText(
    helpText);

  // Make the connections

  connect(mp_discoveriesPage_ui->analysisFilePushButton,
          &QPushButton::clicked,
          this,
          [&]() {
            QString fileName =
              QFileDialog::getSaveFileName(this,
                                           tr("Select the data analysis file"),
                                           QDir::home().absolutePath(),
                                           tr("Any file (*)"),
                                           0,
                                           QFileDialog::DontConfirmOverwrite);

            mp_discoveriesPage_ui->analysisFileLineEdit->setText(fileName);
            mp_discoveriesPage_ui->analysisFileLineEdit->setToolTip(fileName);
          });

  connect(mp_discoveriesPage_ui->addToHistoryPushButton,
          &QPushButton::clicked,
          this,
          [this, plain_text_edit_list_qsp, discoveries_preferences_qsp]() {
            int idx = mp_discoveriesPage_ui->formatsTabWidget->currentIndex();
            QString text = plain_text_edit_list_qsp->at(idx)->toPlainText();

            DataFormatStringSpecif *specif =
              discoveries_preferences_qsp->m_dataFormatStringSpecifHash.value(
                idx);
            if(specif == nullptr)
              {
                qFatal() << "Programming error.";
                return;
              }

            QString historyText = QString("%1:%2").arg(specif->m_label, text);

            mp_discoveriesPage_ui->historyComboBox->addItem(historyText,
                                                            Qt::DisplayRole);
          });


  connect(mp_discoveriesPage_ui->removeFromHistoryPushButton,
          &QPushButton::clicked,
          this,
          [&]() {
            int idx = mp_discoveriesPage_ui->historyComboBox->currentIndex();

            qDebug() << "The current index:" << idx;

            mp_discoveriesPage_ui->historyComboBox->removeItem(idx);
          });

  connect(
    mp_discoveriesPage_ui->historyComboBox,
    static_cast<void (QComboBox::*)(int)>(&QComboBox::activated),
    this,
    [&](int index) {
      // qDebug() << "The activated index:" << index;

      QString historyText =
        mp_discoveriesPage_ui->historyComboBox->itemText(index);

      // qDebug() << "The activated item's text:" << historyText;

      // Extract from the top widget of the discoveries page the two data we
      // need.
      QSharedPointer<DiscoveriesPreferences> discoveries_preferences_qsp =
        nullptr;
      QSharedPointer<QList<QPlainTextEdit *>> plain_text_edit_list_qsp =
        nullptr;

      QVariant variant = mp_discoveriesPage_ui->scrollArea->property(
        "discoveries_preferences_qsp");
      if(variant.canConvert<QSharedPointer<DiscoveriesPreferences>>())
        {
          discoveries_preferences_qsp =
            variant.value<QSharedPointer<DiscoveriesPreferences>>();

          if(discoveries_preferences_qsp.isNull())
            {
              qFatal() << "Programming error.";
              return;
            }
        }
      else
        {
          qFatal() << "Programming error.";
          return;
        }
      variant =
        mp_discoveriesPage_ui->scrollArea->property("plain_text_edit_list_qsp");
      if(variant.canConvert<QSharedPointer<QList<QPlainTextEdit *>>>())
        {
          plain_text_edit_list_qsp =
            variant.value<QSharedPointer<QList<QPlainTextEdit *>>>();

          if(plain_text_edit_list_qsp.isNull())
            {
              qFatal() << "Programming error.";
              return;
            }
        }
      else
        {
          qFatal() << "Programming error.";
          return;
        }

      // What is the label at the beginning of the text that states what
      // is the specialized format string (TIC chrom or Mass spec or Drift
      // spec).

      for(int iter = 0; iter < FormatType::LAST; ++iter)
        {
          // qDebug() << "The discoveries_preferences_qsp:"
          //          << discoveries_preferences_qsp.get();

          // Get the DataFormatStringSpecif instance for iter
          const DataFormatStringSpecif *specif_p =
            discoveries_preferences_qsp->m_dataFormatStringSpecifHash.value(
              iter);

          QString label = specif_p->m_label;

          // qDebug() << "The label:" << label;


          if(historyText.startsWith(label))
            {
              mp_discoveriesPage_ui->formatsTabWidget->setCurrentIndex(iter);

              // Get a pointer to the proper tab plain text editor
              QPlainTextEdit *currentTextEdit =
                plain_text_edit_list_qsp->at(iter);

              // But before setting the text that we got from the combo
              // box, we need to remvove from it the label and the
              // separator that we used in the first place to tag that
              // format string in the history combo box !
              static QRegularExpression regExp(QString("^%1:").arg(label));
              historyText = historyText.remove(regExp);

              currentTextEdit->setPlainText(historyText);

              return;
            }
        }
    });
}

void
ApplicationPreferencesWnd::setupShortcutsPage()
{
  // The management of the actions, via the libXpertMassGui::ActionManager and
  // via the libXpertMassGui::ActionManagerWidget is performed in Application.

  libXpertMassGui::ActionManagerTableWidget *action_manager_table_widget_p =
    new libXpertMassGui::ActionManagerTableWidget(this);

  QIcon svg_icon(":/images/svg/keyboard-keys.svg");

  insertSectionListItem(
    "Shortcuts", svg_icon, action_manager_table_widget_p, Pages::SHORTCUTS);

  // qDebug() << "Done, inserting the section list item.";

  // We now have to fill in the table view.
  // ProgramWindow *program_window_p =
  //   static_cast<Application *>(qApp)->getProgramWindow();
  // if(program_window_p == nullptr)
  //   qFatal() << "The program window is nullptr !!!";

  libXpertMassGui::ActionManager *action_manager_p =
    static_cast<Application *>(qApp)->getProgramWindow()->getActionManager();
  if(action_manager_p == nullptr)
    qFatal() << "The action manager is nullptr !!!";

  action_manager_table_widget_p->setActionManager(action_manager_p);

  action_manager_table_widget_p->fillInTableWithData();

  // int row_count = action_manager_table_widget_p->fillInTableWithData();
  // qDebug() << "Added" << row_count
  //          << "rows in the action manager table widget.";

  // At this point we have filled the table.
}

int
ApplicationPreferencesWnd::insertSectionListItem(const QString &label,
                                                 const QIcon &icon,
                                                 QWidget *section_widget_p,
                                                 int index)
{
  // qDebug() << "Inserting section list item" << label << "at index" << index;

  mp_ui->sectionsListWidget->insertItem(
    index, new QListWidgetItem(icon, label, mp_ui->sectionsListWidget));

  // Returns the index of the inserted widget.
  return mp_ui->sectionsStackedWidget->insertWidget(index, section_widget_p);
}

void
ApplicationPreferencesWnd::apply()
{
  // The very first thing to do is check what is the currently active section.

  int current_index = mp_ui->sectionsStackedWidget->currentIndex();

  switch(current_index)
    {
      case Pages::GENERAL:
        applyGeneralPage();
        break;
      case Pages::SHORTCUTS:
        applyShortcutsPage();
        break;
      case Pages::DISCOVERIES:
        applyDiscoveriesPage();
        break;
      default:
        qFatal() << "Programming error.";
    }
}

void
ApplicationPreferencesWnd::applyGeneralPage()
{
  // qDebug() << "The General page is currently displayed.";

  // The decimal places settings
  libXpertMassCore::MZ_DEC_PLACES =
    mp_decimalPlacesPreferencesWidget_ui->mzSpinBox->value();
  libXpertMassCore::DELTA_MZ_DEC_PLACES =
    mp_decimalPlacesPreferencesWidget_ui->deltaMzSpinBox->value();
  libXpertMassCore::RT_MINUTES_DEC_PLACES =
    mp_decimalPlacesPreferencesWidget_ui->retentionTimeMinutesSpinBox->value();
  libXpertMassCore::RT_SECONDS_DEC_PLACES =
    mp_decimalPlacesPreferencesWidget_ui->retentionTimeSecondsSpinBox->value();
  libXpertMassCore::INTENSITY_DEC_PLACES =
    mp_decimalPlacesPreferencesWidget_ui->intensitySpinBox->value();

  writePeakPickingPreferences();
  writeLowMassDeconvolverPreferences();
}

void
ApplicationPreferencesWnd::writePeakPickingPreferences()
{
  QString settings_file_path =
    static_cast<Application *>(QCoreApplication::instance())
      ->getUserConfigSettingsFilePath();

  QSettings settings(settings_file_path, QSettings::IniFormat);

  settings.beginGroup("PeakPickingPreferences");

  if(mp_peakPickingPreferencesWidget_ui->manualMaxIntensityRadioButton
       ->isChecked())
    settings.setValue("NoiseEstimatorMaxIntensityMode", "MANUAL");
  else if(mp_peakPickingPreferencesWidget_ui
            ->stdDevFactorMaxIntensityRadioButton->isChecked())
    settings.setValue("NoiseEstimatorMaxIntensityMode", "STD_DEV");
  else if(mp_peakPickingPreferencesWidget_ui->percentileMaxIntensityRadioButton
            ->isChecked())
    settings.setValue("NoiseEstimatorMaxIntensityMode", "PERCENTILE");

  settings.setValue(
    "NoiseEstimatorRollingWindowSize",
    mp_peakPickingPreferencesWidget_ui->rollingWindowSizeSpinBox->value());

  settings.setValue("NoiseEstimatorMinimumRequiredElements",
                    mp_peakPickingPreferencesWidget_ui
                      ->mininumRequiredElementsSpinBox->value());

  settings.setValue(
    "NoiseEstimatorHistogramBinCount",
    mp_peakPickingPreferencesWidget_ui->histogramBinCountSpinBox->value());


  settings.setValue("PeakPickerSignalToNoiseRatio",
                    mp_peakPickingPreferencesWidget_ui
                      ->signalToNoiseRatioDoubleSpinBox->value());
  settings.setValue(
    "PeakPickerAllowedMissingPeaks",
    mp_peakPickingPreferencesWidget_ui->missingPeaksCountSpinBox->value());

  settings.sync();

  settings.endGroup();

  qDebug() << "Done writing the peak picking preferences to "
           << settings_file_path;
}

void
ApplicationPreferencesWnd::readPeakPickingPreferences()
{
  QString settings_file_path =
    static_cast<Application *>(QCoreApplication::instance())
      ->getUserConfigSettingsFilePath();

  QSettings settings(settings_file_path, QSettings::IniFormat);

  bool ok = false;

  settings.beginGroup("PeakPickingPreferences");

  QString manual_max_intensity_mode =
    settings.value("NoiseEstimatorMaxIntensityMode", "STD_DEV").toString();
  if(manual_max_intensity_mode == "MANUAL")
    mp_peakPickingPreferencesWidget_ui->manualMaxIntensityRadioButton
      ->setChecked(true);
  else if(manual_max_intensity_mode == "STD_DEV")
    mp_peakPickingPreferencesWidget_ui->stdDevFactorMaxIntensityRadioButton
      ->setChecked(true);
  else if(manual_max_intensity_mode == "PERCENTILE")
    mp_peakPickingPreferencesWidget_ui->percentileMaxIntensityRadioButton
      ->setChecked(true);
  else
    qFatal() << "Programming error.";

  int value = settings.value("NoiseEstimatorRollingWindowSize", 200).toInt(&ok);
  mp_peakPickingPreferencesWidget_ui->rollingWindowSizeSpinBox->setValue(value);

  value =
    settings.value("NoiseEstimatorMinimumRequiredElements", 10).toInt(&ok);
  mp_peakPickingPreferencesWidget_ui->mininumRequiredElementsSpinBox->setValue(
    value);

  value = settings.value("NoiseEstimatorHistogramBinCount", 30).toInt(&ok);
  mp_peakPickingPreferencesWidget_ui->histogramBinCountSpinBox->setValue(value);

  value = settings.value("PeakPickerSignalToNoiseRatio", 1).toInt(&ok);
  mp_peakPickingPreferencesWidget_ui->signalToNoiseRatioDoubleSpinBox->setValue(
    value);

  value = settings.value("PeakPickerAllowedMissingPeaks", 0).toInt(&ok);
  mp_peakPickingPreferencesWidget_ui->missingPeaksCountSpinBox->setValue(value);

  settings.endGroup();
}

std::tuple<pappso::LocalSignalToNoiseEstimator::Parameters,
           pappso::HighResPeakPicker::Parameters>
ApplicationPreferencesWnd::getPeakPickingPreferences() const
{
  pappso::LocalSignalToNoiseEstimator::Parameters noise_estimator_params;

  if(mp_peakPickingPreferencesWidget_ui->manualMaxIntensityRadioButton
       ->isChecked())
    {
      noise_estimator_params.maxIntMode =
        pappso::LocalSignalToNoiseEstimator::MANUAL;
      noise_estimator_params.maxIntensity =
        mp_peakPickingPreferencesWidget_ui->manualMaxIntensitySpinBox->value();
    }
  else if(mp_peakPickingPreferencesWidget_ui
            ->stdDevFactorMaxIntensityRadioButton->isChecked())
    {
      noise_estimator_params.maxIntMode =
        pappso::LocalSignalToNoiseEstimator::AUTOMAXBYSTDEV;
      noise_estimator_params.maxIntStdDevFactor =
        mp_peakPickingPreferencesWidget_ui->stdDevFactorMaxIntensitySpinBox
          ->value();
    }
  else if(mp_peakPickingPreferencesWidget_ui
            ->stdDevFactorMaxIntensityRadioButton->isChecked())
    {
      noise_estimator_params.maxIntMode =
        pappso::LocalSignalToNoiseEstimator::AUTOMAXBYPERCENTILE;
      noise_estimator_params.maxIntPercentile =
        mp_peakPickingPreferencesWidget_ui->percentileMaxIntensitySpinBox
          ->value();
    }
  else
    qFatal() << "Programming error.";

  noise_estimator_params.windowSize =
    mp_peakPickingPreferencesWidget_ui->rollingWindowSizeSpinBox->value();
  noise_estimator_params.binCount =
    mp_peakPickingPreferencesWidget_ui->histogramBinCountSpinBox->value();
  noise_estimator_params.minRequiredElements =
    mp_peakPickingPreferencesWidget_ui->mininumRequiredElementsSpinBox->value();


  pappso::HighResPeakPicker::Parameters peak_picker_params;

  peak_picker_params.signalToNoise =
    mp_peakPickingPreferencesWidget_ui->signalToNoiseRatioDoubleSpinBox
      ->value();

  peak_picker_params.missingPeakCount =
    mp_peakPickingPreferencesWidget_ui->missingPeaksCountSpinBox->value();

  return std::tuple<pappso::LocalSignalToNoiseEstimator::Parameters,
                    pappso::HighResPeakPicker::Parameters>(
    noise_estimator_params, peak_picker_params);
}

void
ApplicationPreferencesWnd::writeLowMassDeconvolverPreferences()
{
  QString settings_file_path =
    static_cast<Application *>(QCoreApplication::instance())
      ->getUserConfigSettingsFilePath();

  QSettings settings(settings_file_path, QSettings::IniFormat);

  settings.remove("LowMassDeconvolverPreferences");

  settings.beginGroup("LowMassDeconvolverPreferences");

  foreach(const QString &params_title,
          m_lowMassDeconvolverParameterPairs.keys())
    {
      settings.beginGroup(params_title);

      libXpertMassCore::LowMassDeconvolver::Parameters &params =
        m_lowMassDeconvolverParameterPairs[params_title];

      settings.setValue("MinCharge", params.minCharge);
      settings.setValue("MaxCharge", params.maxCharge);
      settings.setValue("PpmMassOrMzTolerance", params.ppmMassTolerance);
      settings.setValue("MinMass", params.minMass);
      settings.setValue("MaxMass", params.maxMass);
      settings.setValue("MinIntensity", params.minIntensity);
      settings.setValue(
        "AveragineFormula",
        params.averagineFormula.getActionFormula(true /*with_title*/));
      settings.setValue("IsotopicDataFilePath", params.isotopicDataFilePath);
      settings.setValue("ClusterShapeTolerance", params.clusterShapeTolerance);
      settings.setValue("MinScore", params.minScore);

      settings.endGroup();
    }

  settings.sync();

  settings.endGroup();

  qDebug() << "Done writing the low mass deconvolver preferences to "
           << settings_file_path;
}

void
ApplicationPreferencesWnd::readLowMassDeconvolverPreferences()
{
  QSettings settings(static_cast<Application *>(QCoreApplication::instance())
                       ->getUserConfigSettingsFilePath(),
                     QSettings::IniFormat);

  bool ok = false;

  settings.beginGroup("LowMassDeconvolverPreferences");

  QStringList sub_groups = settings.childGroups();

  qDebug() << "There are" << sub_groups.size() << "sub groups.";

  for(const QString &sub_group : sub_groups)
    {
      qDebug() << "Found sub-group:" << sub_group;

      settings.beginGroup(sub_group);

      int minCharge = settings.value("MinCharge", 1).toInt(&ok);
      Q_ASSERT(ok);
      int maxCharge = settings.value("MaxCharge", 1).toInt(&ok);
      Q_ASSERT(ok);

      double minMass = settings.value("MinMass", 150).toDouble(&ok);
      Q_ASSERT(ok);
      double maxMass = settings.value("MaxMass", 5000).toDouble(&ok);
      Q_ASSERT(ok);

      double ppmMassTolerance =
        settings.value("PpmMassOrMzTolerance", 10).toDouble(&ok);
      Q_ASSERT(ok);

      double minIntensity = settings.value("MinIntensity", 0.05).toDouble(&ok);
      Q_ASSERT(ok);

      QString isotopicDataFilePath =
        settings.value("IsotopicDataFilePath", "").toString();

      QString averagineFormula =
        settings.value("AveragineFormula", "").toString();

      double clusterShapeTolerance =
        settings.value("ClusterShapeTolerance", 0.7).toDouble(&ok);
      Q_ASSERT(ok);
      double minScore = settings.value("MinScore", 0.85).toDouble(&ok);
      Q_ASSERT(ok);

      libXpertMassCore::LowMassDeconvolver::Parameters params(
        minCharge,
        maxCharge,
        minMass,
        maxMass,
        ppmMassTolerance,
        minIntensity,
        isotopicDataFilePath,
        averagineFormula,
        clusterShapeTolerance,
        minScore);

      m_lowMassDeconvolverParameterPairs[sub_group] = params;
      qDebug() << "Now updating settings for sub group:" << sub_group
               << "with params:" << params.toString();

      settings.endGroup();
    }

  settings.endGroup();

  populateLowMassDeconvolverPreferencesCombobox();
}

bool
ApplicationPreferencesWnd::getLowMassDeconvolverPreferences(
  libXpertMassCore::LowMassDeconvolver::Parameters &params) const
{
  QString file_name =
    mp_lowMassDeconvolverPreferencesWidget_ui->isotopicDataFileLineEdit->text();
  QFileInfo file_info(file_name);
  if(!file_info.exists())
    {
      QMessageBox::warning(const_cast<ApplicationPreferencesWnd *>(this),
                           "Low mass deconvolver preference settings",
                           "Please select an isotopic data file",
                           QMessageBox::Ok);
      return false;
    }

  params.isotopicDataFilePath = file_name;

  QString averagine_formula_string =
    mp_lowMassDeconvolverPreferencesWidget_ui->averagineLineEdit->text();
  if(averagine_formula_string.isEmpty())
    {
      QMessageBox::warning(
        const_cast<ApplicationPreferencesWnd *>(this),
        "Low mass deconvolver preference settings",
        "Please fill in a correct neutral Averagine formula\n"
        "like C4.29H6.86N0.86O2.00",
        QMessageBox::Ok);
      return false;
    }
  params.averagineFormula = libXpertMassCore::Formula(averagine_formula_string);

  params.minCharge =
    mp_lowMassDeconvolverPreferencesWidget_ui->minChargeSpinBox->value();

  params.maxCharge =
    mp_lowMassDeconvolverPreferencesWidget_ui->maxChargeSpinBox->value();

  params.minMass =
    mp_lowMassDeconvolverPreferencesWidget_ui->minMassSpinBox->value();

  params.maxMass =
    mp_lowMassDeconvolverPreferencesWidget_ui->maxMassSpinBox->value();

  params.minIntensity = mp_lowMassDeconvolverPreferencesWidget_ui
                          ->minIntensityDoubleSpinBox->value();

  params.ppmMassTolerance = mp_lowMassDeconvolverPreferencesWidget_ui
                              ->massToleranceDoubleSpinBox->value();

  params.clusterShapeTolerance =
    mp_lowMassDeconvolverPreferencesWidget_ui
      ->relativeClusterShapeToleranceDoubleSpinBox->value();

  params.minScore =
    mp_lowMassDeconvolverPreferencesWidget_ui->minScoreDoubleSpinBox->value();

  return true;
}

int
ApplicationPreferencesWnd::populateLowMassDeconvolverPreferencesCombobox()
{
  if(!m_lowMassDeconvolverParameterPairs.size())
    return 0;

  int count = 0;
  foreach(const QString &params_title,
          m_lowMassDeconvolverParameterPairs.keys())
    {
      mp_lowMassDeconvolverPreferencesWidget_ui->historyItemTitleComboBox
        ->addItem(params_title);
      ++count;
    }

  qDebug() << "Effectively added" << count << "history items.";

  populateLowMassDeconvolverPreferencesForHistoryItem(
    m_lowMassDeconvolverParameterPairs.keys().last());

  return count;
}

bool
ApplicationPreferencesWnd::populateLowMassDeconvolverPreferencesForHistoryItem(
  const QString &params_title)
{
  mp_lowMassDeconvolverPreferencesWidget_ui->historyItemTitleComboBox
    ->setCurrentText(params_title);

  libXpertMassCore::LowMassDeconvolver::Parameters params =
    m_lowMassDeconvolverParameterPairs.value(params_title);

  mp_lowMassDeconvolverPreferencesWidget_ui->minChargeSpinBox->setValue(
    params.minCharge);

  mp_lowMassDeconvolverPreferencesWidget_ui->maxChargeSpinBox->setValue(
    params.maxCharge);

  mp_lowMassDeconvolverPreferencesWidget_ui->minMassSpinBox->setValue(
    params.minMass);

  mp_lowMassDeconvolverPreferencesWidget_ui->maxMassSpinBox->setValue(
    params.maxMass);

  mp_lowMassDeconvolverPreferencesWidget_ui->minIntensityDoubleSpinBox
    ->setValue(params.minIntensity);

  mp_lowMassDeconvolverPreferencesWidget_ui->massToleranceDoubleSpinBox
    ->setValue(params.ppmMassTolerance);

  mp_lowMassDeconvolverPreferencesWidget_ui->isotopicDataFileLineEdit->setText(
    params.isotopicDataFilePath);

  mp_lowMassDeconvolverPreferencesWidget_ui->averagineLineEdit->setText(
    params.averagineFormula.getActionFormula(true /*with_title*/));

  mp_lowMassDeconvolverPreferencesWidget_ui
    ->relativeClusterShapeToleranceDoubleSpinBox->setValue(
      params.clusterShapeTolerance);

  mp_lowMassDeconvolverPreferencesWidget_ui->minScoreDoubleSpinBox->setValue(
    params.minScore);

  return true;
}

bool
ApplicationPreferencesWnd::
  updateLowMassDeconvolverPreferencesCurrentHistoryItem()
{
  QString current_text = mp_lowMassDeconvolverPreferencesWidget_ui
                           ->historyItemTitleComboBox->currentText();

  if(current_text.isEmpty())
    {
      QMessageBox::warning(const_cast<ApplicationPreferencesWnd *>(this),
                           "Low mass deconvolver preference settings",
                           "Please enter a name for the current parameter set",
                           QMessageBox::Ok);
      return false;
    }

  libXpertMassCore::LowMassDeconvolver::Parameters &params =
    m_lowMassDeconvolverParameterPairs[current_text];

  QString file_name =
    mp_lowMassDeconvolverPreferencesWidget_ui->isotopicDataFileLineEdit->text();
  QFileInfo file_info(file_name);
  if(!file_info.exists())
    {
      QMessageBox::warning(const_cast<ApplicationPreferencesWnd *>(this),
                           "Low mass deconvolver preference settings",
                           "Please select an isotopic data file",
                           QMessageBox::Ok);
      return false;
    }

  params.isotopicDataFilePath = file_name;

  QString averagine_formula_string =
    mp_lowMassDeconvolverPreferencesWidget_ui->averagineLineEdit->text();
  if(averagine_formula_string.isEmpty())
    {
      QMessageBox::warning(
        const_cast<ApplicationPreferencesWnd *>(this),
        "Low mass deconvolver preference settings",
        "Please fill in a correct neutral Averagine formula\n"
        "like C4.29H6.86N0.86O2.00",
        QMessageBox::Ok);
      return false;
    }
  params.averagineFormula = libXpertMassCore::Formula(averagine_formula_string);

  params.minCharge =
    mp_lowMassDeconvolverPreferencesWidget_ui->minChargeSpinBox->value();

  params.maxCharge =
    mp_lowMassDeconvolverPreferencesWidget_ui->maxChargeSpinBox->value();

  params.minMass =
    mp_lowMassDeconvolverPreferencesWidget_ui->minMassSpinBox->value();

  params.maxMass =
    mp_lowMassDeconvolverPreferencesWidget_ui->maxMassSpinBox->value();

  params.minIntensity = mp_lowMassDeconvolverPreferencesWidget_ui
                          ->minIntensityDoubleSpinBox->value();

  params.ppmMassTolerance = mp_lowMassDeconvolverPreferencesWidget_ui
                              ->massToleranceDoubleSpinBox->value();
  params.clusterShapeTolerance =
    mp_lowMassDeconvolverPreferencesWidget_ui
      ->relativeClusterShapeToleranceDoubleSpinBox->value();

  params.minScore =
    mp_lowMassDeconvolverPreferencesWidget_ui->minScoreDoubleSpinBox->value();

  return true;
}

void
ApplicationPreferencesWnd::addLowMassDeconvolverPreferencesToHistory()
{
  libXpertMassCore::LowMassDeconvolver::Parameters params;

  bool ok = getLowMassDeconvolverPreferences(params);

  if(!ok)
    {
      QMessageBox::warning(const_cast<ApplicationPreferencesWnd *>(this),
                           "Low mass deconvolver preference settings",
                           "Failed to get the preferences",
                           QMessageBox::Ok);
      return;
    }

  QString current_text = mp_lowMassDeconvolverPreferencesWidget_ui
                           ->historyItemTitleComboBox->currentText();
  if(current_text.isEmpty())
    current_text = QDateTime::currentDateTime().toString();

  mp_lowMassDeconvolverPreferencesWidget_ui->historyItemTitleComboBox->addItem(
    current_text);

  m_lowMassDeconvolverParameterPairs[current_text] = params;

  writeLowMassDeconvolverPreferences();
}

void
ApplicationPreferencesWnd::removeLowMassDeconvolverPreferencesFromHistory()
{

  Q_ASSERT(mp_lowMassDeconvolverPreferencesWidget_ui->historyItemTitleComboBox
             ->count() == m_lowMassDeconvolverParameterPairs.size());

  if(!mp_lowMassDeconvolverPreferencesWidget_ui->historyItemTitleComboBox
        ->count())
    return;

  QString current_text = mp_lowMassDeconvolverPreferencesWidget_ui
                           ->historyItemTitleComboBox->currentText();

  mp_lowMassDeconvolverPreferencesWidget_ui->historyItemTitleComboBox
    ->removeItem(mp_lowMassDeconvolverPreferencesWidget_ui
                   ->historyItemTitleComboBox->currentIndex());

  qDebug() << "Before removing, items:"
           << m_lowMassDeconvolverParameterPairs.size();

  m_lowMassDeconvolverParameterPairs.remove(current_text);

  qDebug() << "After removing, items:"
           << m_lowMassDeconvolverParameterPairs.size();

  Q_ASSERT(mp_lowMassDeconvolverPreferencesWidget_ui->historyItemTitleComboBox
             ->count() == m_lowMassDeconvolverParameterPairs.size());

  writeLowMassDeconvolverPreferences();

  if(!mp_lowMassDeconvolverPreferencesWidget_ui->historyItemTitleComboBox
        ->count())
    return;

  current_text = mp_lowMassDeconvolverPreferencesWidget_ui
                   ->historyItemTitleComboBox->currentText();

  if(!populateLowMassDeconvolverPreferencesForHistoryItem(current_text))
    {
      QMessageBox::warning(const_cast<ApplicationPreferencesWnd *>(this),
                           "Low mass deconvolver preference settings",
                           "Failed to update the preferences",
                           QMessageBox::Ok);
    }
}


void
ApplicationPreferencesWnd::applyShortcutsPage()
{
  qDebug() << "The Shortcuts page is currently displayed.";

  // At this point, all we have to do is save the shortcuts.

  libXpertMassGui::ActionManager *action_manager_p =
    static_cast<Application *>(qApp)->getProgramWindow()->getActionManager();
  if(action_manager_p == nullptr)
    {
      qFatal() << "The action manager is nullptr !!!";
      return;
    }

  action_manager_p->saveActionData();
}

void
ApplicationPreferencesWnd::applyDiscoveriesPage()
{
  // qDebug() << "The Discoveries page is currently displayed.";

  // Extract from the top widget of the discoveries page the two data we need.
  QSharedPointer<DiscoveriesPreferences> discoveries_preferences_qsp = nullptr;
  QSharedPointer<QList<QPlainTextEdit *>> plain_text_edit_list_qsp   = nullptr;

  QVariant variant =
    mp_discoveriesPage_ui->scrollArea->property("discoveries_preferences_qsp");
  if(variant.canConvert<QSharedPointer<DiscoveriesPreferences>>())
    {
      discoveries_preferences_qsp =
        variant.value<QSharedPointer<DiscoveriesPreferences>>();

      if(discoveries_preferences_qsp.isNull())
        {
          qFatal() << "Programming error.";
          return;
        }
    }
  else
    {
      qFatal() << "Programming error.";
      return;
    }
  variant =
    mp_discoveriesPage_ui->scrollArea->property("plain_text_edit_list_qsp");
  if(variant.canConvert<QSharedPointer<QList<QPlainTextEdit *>>>())
    {
      plain_text_edit_list_qsp =
        variant.value<QSharedPointer<QList<QPlainTextEdit *>>>();

      if(plain_text_edit_list_qsp.isNull())
        {
          qFatal() << "Programming error.";
          return;
        }
    }
  else
    {
      qFatal() << "Programming error.";
      return;
    }

  // Start by setting the value to a ground state.
  discoveries_preferences_qsp->m_recordTarget = RecordTarget::RECORD_NOTHING;

  if(mp_discoveriesPage_ui->toConsoleCheckBox->isChecked())
    discoveries_preferences_qsp->m_recordTarget |=
      RecordTarget::RECORD_TO_CONSOLE;

  if(mp_discoveriesPage_ui->toClipboardCheckBox->isChecked())
    {
      discoveries_preferences_qsp->m_recordTarget |=
        RecordTarget::RECORD_TO_CLIPBOARD;

      QMessageBox::StandardButton ret;
      QString msgBoxTitle =
        QString("%1 - Record to clipboard").arg(m_applicationName);
      ret = QMessageBox::warning(this,
                                 msgBoxTitle,
                                 tr("Initialize the clipboard by emptying it?"),
                                 QMessageBox::Yes | QMessageBox::No);

      if(ret == QMessageBox::Yes)
        {
          QClipboard *clipboard = QApplication::clipboard();
          clipboard->setText("", QClipboard::Clipboard);
        }
    }

  if(mp_discoveriesPage_ui->toFileGroupBox->isChecked())
    discoveries_preferences_qsp->m_recordTarget |= RecordTarget::RECORD_TO_FILE;

  // qDebug() << "Record target: " <<
  // discoveries_preferences_qsp->m_recordTarget ;

  // And now if the record should be directed to a file, make the
  // verifications that we can open the files in either write only or
  // append mode.

  if((discoveries_preferences_qsp->m_recordTarget &
      RecordTarget::RECORD_TO_FILE) == RecordTarget::RECORD_TO_FILE)
    {
      // qDebug() << "Record to file bit set.";

      discoveries_preferences_qsp->m_fileName =
        mp_discoveriesPage_ui->analysisFileLineEdit->text();
      QFile analysisFile(discoveries_preferences_qsp->m_fileName);

      if(mp_discoveriesPage_ui->overwriteRadioButton->isChecked())
        {
          // qDebug() << "overwrite checked";

          discoveries_preferences_qsp->m_fileOpenMode = QIODevice::Truncate;

          // Interestingly, only truncate did not work. Since writeonly
          // implies
          // truncate, that's fine, and it works.

          analysisFile.open(QIODevice::WriteOnly);

          analysisFile.close();
        }
      else
        {
          // qDebug() << "append checked";

          discoveries_preferences_qsp->m_fileOpenMode = QIODevice::Append;

          analysisFile.open(QIODevice::Append);
          analysisFile.close();
        }
    }

  // At this point we need to get all the format strings from the various
  // tab pages.

  for(int iter = 0; iter < FormatType::LAST; ++iter)
    {
      QPlainTextEdit *textEdit = plain_text_edit_list_qsp->at(iter);
      QString formatString     = textEdit->toPlainText();

      // We need to get the DataFormatStringSpecif instance for the current
      // iter. We assume that the member m_discoveriesPreferences instance has
      // been correctly setup.

      DataFormatStringSpecif *specif =
        discoveries_preferences_qsp->m_dataFormatStringSpecifHash.value(iter);

      // Now modify the format string.
      specif->m_format = formatString;

      // qDebug() << "Dealt with iter " << iter << "with format string:" <<
      // specif->m_format;
    }

  // Store the history to the settings, alongside the geometry of the window.

  QStringList historyList;

  for(int iter = 0; iter < mp_discoveriesPage_ui->historyComboBox->count();
      ++iter)
    historyList.append(mp_discoveriesPage_ui->historyComboBox->itemText(iter));

  QString settings_file_path =
    static_cast<Application *>(QCoreApplication::instance())
      ->getUserConfigSettingsFilePath();

  QSettings settings(settings_file_path, QSettings::IniFormat);

  settings.beginGroup("DiscoveriesPreferences");

  settings.setValue("destSplitter",
                    mp_discoveriesPage_ui->destSplitter->saveState());
  settings.setValue("formatSplitter",
                    mp_discoveriesPage_ui->formatSplitter->saveState());

  settings.setValue("history", historyList);
  settings.setValue("geometry", saveGeometry());

  // And now store the different format strings, if there are any, so that
  // upon reopening, it won't necessarily be required to set them again and
  // again.

  for(int iter = 0; iter < FormatType::LAST; ++iter)
    {
      const DataFormatStringSpecif *specif =
        discoveries_preferences_qsp->m_dataFormatStringSpecifHash.value(iter);

      if(specif == nullptr)
        {
          qFatal() << "Programming error.";
          return;
        }

      QString tabString   = specif->m_label;
      QString settingsKey = QString("%1_formatString").arg(tabString);

      QPlainTextEdit *textEdit = static_cast<QPlainTextEdit *>(
        mp_discoveriesPage_ui->formatsTabWidget->widget(iter));

      QString settingsValue = textEdit->toPlainText();

      settings.setValue(settingsKey, settingsValue);

      // qDebug() << "Writing setting key/value for iter"
      //<< iter << settingsKey << settingsValue;
    }

  settings.sync();

  settings.endGroup();

  qDebug() << "Done writing discoveries preferences to " << settings_file_path;

  // ProgramWindow *program_window_p =
  //   static_cast<Application *>(qApp)->getProgramWindow();
  // if(program_window_p == nullptr)
  //   qFatal() << "The program window is nullptr !!!";

  static_cast<Application *>(qApp)
    ->getProgramWindow()
    ->updateDiscoveriesPreferences(*discoveries_preferences_qsp);
}

} // namespace MineXpert
} // namespace MsXpS
