/*
 * This program source code file is part of KiCad, a free EDA CAD application.
 *
 * Copyright (C) 2018 Jean-Pierre Charras jp.charras at wanadoo.fr
 * Copyright The KiCad Developers, see AUTHORS.txt for contributors.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you may find one here:
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * or you may search the http://www.gnu.org website for the version 2 license,
 * or you may write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */

#pragma once

#include <board_connected_item.h>
#include <eda_shape.h>


class LINE_READER;
class EDA_DRAW_FRAME;
class FOOTPRINT;
class MSG_PANEL_ITEM;


class PCB_SHAPE : public BOARD_CONNECTED_ITEM, public EDA_SHAPE
{
public:
    PCB_SHAPE( BOARD_ITEM* aParent, KICAD_T aItemType, SHAPE_T aShapeType );

    PCB_SHAPE( BOARD_ITEM* aParent = nullptr, SHAPE_T aShapeType = SHAPE_T::SEGMENT );

    // Do not create a copy constructor & operator=.
    // The ones generated by the compiler are adequate.

    ~PCB_SHAPE() override;

    void CopyFrom( const BOARD_ITEM* aOther ) override;

    static bool ClassOf( const EDA_ITEM* aItem )
    {
        return aItem && PCB_SHAPE_T == aItem->Type();
    }

    wxString GetClass() const override
    {
        return wxT( "PCB_SHAPE" );
    }

    void Serialize( google::protobuf::Any &aContainer ) const override;
    bool Deserialize( const google::protobuf::Any &aContainer ) override;

    bool IsConnected() const override;

    wxString GetFriendlyName() const override { return getFriendlyName(); }

    bool IsType( const std::vector<KICAD_T>& aScanTypes ) const override;

    void SetLayer( PCB_LAYER_ID aLayer ) override;
    PCB_LAYER_ID GetLayer() const override { return m_layer; }

    bool IsOnLayer( PCB_LAYER_ID aLayer ) const override;

    virtual LSET GetLayerSet() const override;
    virtual void SetLayerSet( const LSET& aLayers ) override;

    void SetPosition( const VECTOR2I& aPos ) override { setPosition( aPos ); }
    VECTOR2I GetPosition() const override { return getPosition(); }

    VECTOR2I GetCenter() const override { return getCenter(); }

    /**
     * @return a list of connection points (may be empty): points where this shape can form
     * electrical connections to other shapes that are natural "start/end" points.
     */
    std::vector<VECTOR2I> GetConnectionPoints() const;

    bool HasLineStroke() const override { return true; }

    STROKE_PARAMS GetStroke() const override { return m_stroke; }
    void SetStroke( const STROKE_PARAMS& aStroke ) override { m_stroke = aStroke; }

    int GetWidth() const override;

    void StyleFromSettings( const BOARD_DESIGN_SETTINGS& settings ) override;

    /**
     * Return 4 corners for a rectangle or rotated rectangle (stored as a poly).  Unimplemented
     * for other shapes.
     */
    virtual std::vector<VECTOR2I> GetCorners() const;

    /**
     * Allows items to return their visual center rather than their anchor. For some shapes this
     * is similar to GetCenter(), but for unfilled shapes a point on the outline is better.
     */
    const VECTOR2I GetFocusPosition() const override;

    /**
     * Make a set of SHAPE objects representing the PCB_SHAPE.  Caller owns the objects.
     */
    std::shared_ptr<SHAPE> GetEffectiveShape( PCB_LAYER_ID aLayer = UNDEFINED_LAYER,
                                              FLASHING aFlash = FLASHING::DEFAULT ) const override;

    bool IsProxyItem() const override { return m_proxyItem; }
    void SetIsProxyItem( bool aIsProxy = true ) override;

    void GetMsgPanelInfo( EDA_DRAW_FRAME* aFrame, std::vector<MSG_PANEL_ITEM>& aList ) override;

    const BOX2I GetBoundingBox() const override { return getBoundingBox(); }

    bool HitTest( const VECTOR2I& aPosition, int aAccuracy = 0 ) const override
    {
        return hitTest( aPosition, aAccuracy );
    }

    bool HitTest( const BOX2I& aRect, bool aContained, int aAccuracy = 0 ) const override
    {
        return hitTest( aRect, aContained, aAccuracy );
    }

    void Normalize() override;

    /**
     * Normalize coordinates to compare 2 similar PCB_SHAPES
     * similat to Normalize(), but also normalize SEGMENT end points
     * needed only for graphic comparisons
     */
    void NormalizeForCompare() override;

    void Move( const VECTOR2I& aMoveVector ) override;

    void Rotate( const VECTOR2I& aRotCentre, const EDA_ANGLE& aAngle ) override;

    void Flip( const VECTOR2I& aCentre, FLIP_DIRECTION aFlipDirection ) override;

    virtual void Mirror( const VECTOR2I& aCentre, FLIP_DIRECTION aFlipDirection ) override;

    void Scale( double aScale );

    /**
     * Convert the shape to a closed polygon.  Circles and arcs are approximated by segments.
     *
     * @param aBuffer is a buffer to store the polygon.
     * @param aClearance is the clearance around the pad.
     * @param aError is the maximum deviation from a true arc.
     * @param aErrorLoc whether any approximation error should be placed inside or outside
     * @param ignoreLineWidth is used for edge cut items where the line width is only for
     *                        visualization
     */
    void TransformShapeToPolygon( SHAPE_POLY_SET& aBuffer, PCB_LAYER_ID aLayer, int aClearance,
                                  int aError, ERROR_LOC aErrorLoc,
                                  bool ignoreLineWidth = false ) const override;

    wxString GetItemDescription( UNITS_PROVIDER* aUnitsProvider, bool aFull ) const override;

    BITMAPS GetMenuImage() const override;

    EDA_ITEM* Clone() const override;

    const BOX2I ViewBBox() const override;

    std::vector<int> ViewGetLayers() const override;

    ///< @copydoc VIEW_ITEM::ViewGetLOD
    double ViewGetLOD( int aLayer, const KIGFX::VIEW* aView ) const override;

    double Similarity( const BOARD_ITEM& aBoardItem ) const override;

    bool operator==( const PCB_SHAPE& aShape ) const;
    bool operator==( const BOARD_ITEM& aBoardItem ) const override;

    void SetHasSolderMask( bool aVal ) { m_hasSolderMask = aVal; }
    bool HasSolderMask() const         { return m_hasSolderMask; }

    void SetLocalSolderMaskMargin( std::optional<int> aMargin ) { m_solderMaskMargin = aMargin; }
    std::optional<int> GetLocalSolderMaskMargin() const         { return m_solderMaskMargin; }

    int GetSolderMaskExpansion() const;

#if defined(DEBUG)
    void Show( int nestLevel, std::ostream& os ) const override { ShowDummy( os ); }
#endif

protected:
    void swapData( BOARD_ITEM* aImage ) override;

    int getMaxError() const override;

    struct cmp_drawings
    {
        bool operator()( const BOARD_ITEM* aFirst, const BOARD_ITEM* aSecond ) const;
    };

    bool               m_hasSolderMask;
    std::optional<int> m_solderMaskMargin;
};
