/*
 * Copyright (c) 2022, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */


/*
 * @test
 * @bug 8290126
 * @summary Add a check in JavadocTester for "javadoc should not crash"
 * @library /tools/lib/ ../lib
 * @modules jdk.javadoc/jdk.javadoc.internal.tool
 * @build toolbox.ToolBox javadoc.tester.* TestJavadocTester
 * @run main TestJavadocTesterCrash
 */

import com.sun.source.doctree.DocTree;
import jdk.javadoc.doclet.Taglet;

import javax.lang.model.element.Element;
import java.nio.file.Path;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;

/**
 * Tests that {@code JavadocTester} detects and reports exceptions.
 *
 * It is not a direct test of the javadoc tool or the output generated by the
 * Standard Doclet, although both are indirectly used as part of this test.
 *
 * The test uses the infrastructure of TestJavadocTester, but cannot be
 * added to the tests there, which rely on checking aspects of the output
 * from a single run on javadoc.  This test forces a crash to occur in
 * javadoc, and verifies that JavadocTester detects and reports the crash.
 *
 * Arguably, a crash in a user-provided taglet should not cause a full stack
 * trace. If that is ever fixed, we would need to revisit the goals and mechanism
 * of this test.
 */
public class TestJavadocTesterCrash extends TestJavadocTester {
    public static void main(String... args) throws Exception {
        var tester = new TestJavadocTesterCrash();
        tester.runTests();
    }

    /** A taglet that can throw an exception. */
    public static class TestTaglet implements Taglet {
        public TestTaglet() { }

        @Override
        public Set<Location> getAllowedLocations() {
            return EnumSet.allOf(Location.class);
        }

        @Override
        public boolean isInlineTag() {
            return true;
        }

        @Override
        public String getName() {
            return "test-taglet";
        }

        @Override
        public String toString(List<? extends DocTree> tags, Element element) {
            String s = tags.toString();
            if (s.contains("test")) {
                throw new Error("demo error");
            }
            return s;
        }
    }

    @Test
    public void testDetectException(Path base) throws Exception {
        messages.clear();
        Path src = base.resolve("src");
        tb.writeJavaFiles(src,
                """
                    package p;
                    /** Comment.
                     abc {@test-taglet simple} {@test-taglet test} xyz
                     */
                    public class C { }""");
        javadoc("-d", base.resolve("api").toString(),
                "-tagletpath", System.getProperty("test.class.path"),
                "-taglet", TestTaglet.class.getName(),
                "-sourcepath", src.toString(),
                "p");
        checkExit(Exit.ERROR);

        // verify that the taglet threw an exception as intended
        new OutputChecker(Output.OUT)
                .setExpectOrdered(true)
                .check("Generating testDetectException/api/p/C.html...",
                        "error: An internal exception has occurred.",
                        "(java.lang.Error: demo error)",
                        "1 error");

        // verify that JavadocTester detected the crash
        checkMessages("""
            FAILED: STDERR: output not as expected
            >> the following text was found:""");
    }
}
