use core:io;

/**
 * Read serialized objects and output a human-readable string representation. This does not require
 * that we have access to all objects stored in the serialized stream.
 */
class TextObjStream {
	// Create.
	init(IStream src) {
		init() { src = src; }
	}

	// Read an object to a StrBuf.
	void read(StrBuf to) {
		objects.clear();

		if (bad) {
			to << "<bad stream>";
		} else {
			Nat type = Nat:read(src);
			read(to, type);
		}
	}

	// Read to a string.
	Str read() {
		StrBuf to;
		read(to);
		to.toS();
	}

private:

	/**
	 * Type description.
	 */
	class Type {
		Str name;
		TypeInfo flags;
		Nat parent;
		Member[] members;

		init(Str name, TypeInfo flags, Nat parent) {
			init() {
				name = name;
				flags = flags;
				parent = parent;
			}
		}

		Str beginSym() {
			if (flags & TypeInfo:tuple)
				"[";
			else if (flags & TypeInfo:maybe)
				":";
			else
				"{";
		}

		Str endSym() {
			if (flags & TypeInfo:tuple)
				"]";
			else if (flags & TypeInfo:maybe)
				"";
			else
				"}";
		}
	}

	/**
	 * Member description.
	 */
	value Member {
		Nat type;
		Str name;

		init(Nat type, Str name) {
			init() {
				type = type;
				name = name;
			}
		}
	}

	// Source stream.
	IStream src;

	// Remember all types.
	Nat->Type types;

	// Remember previously stored objects.
	Set<Nat> objects;

	// Did we encounter something bad, and cannot continue reading?
	Bool bad;

	// Read an object of a known type.
	void read(StrBuf to, Nat type) {
		// Stop reading if the stream is bad.
		if (bad)
			return;

		if (readPrimitive(to, type))
			return;

		Type t = findType(type, to);
		if (bad)
			return;

		if (t.flags & TypeInfo:classType) {
			readObject(to, t);
		} else {
			readValue(to, t);
		}
	}

	// Read an object.
	void readObject(StrBuf to, Type type) {
		// A previous object?
		Nat objId = Nat:read(src);
		if (objects.has(objId)) {
			to << "<link to instance " << objId << ">";
			return;
		}
		objects.put(objId);

		// Actual type?
		Nat actualType = Nat:read(src);
		Type actual = findType(actualType, to);
		if (bad)
			return;

		to << actual.name << " (instance " << objId << ") ";
		readMembers(to, actual);
	}

	// Read a value.
	void readValue(StrBuf to, Type type) {
		to << type.name << " ";
		readMembers(to, type);
	}

	// Read members from a type.
	void readMembers(StrBuf to, Type type) {
		readMembers(to, type, null);
	}

	void readMembers(StrBuf to, Nat typeId, Type? parent) {
		readMembers(to, findType(typeId, to), parent);
	}

	void readMembers(StrBuf to, Type type, Type? parent) {
		if (bad)
			return;

		Bool brackets = true;
		if (parent) {
			if (parent.beginSym() == type.beginSym())
				brackets = false;
		}

		if (brackets) {
			to << type.beginSym() << "\n";
			to.indent();
		}

		if (type.parent != 0)
			readMembers(to, type.parent, type);

		if (type.flags & TypeInfo:tuple) {
			Nat count = Nat:read(src);
			for (Nat i = 0; i < count; i++) {
				if (type.members.count > 1)
					to << "(";

				for (i, m in type.members) {
					if (i > 0)
						to << ", ";
					read(to, m.type);
				}

				if (type.members.count > 1)
					to << ")";
				to << "\n";
			}
		} else if (type.flags & TypeInfo:maybe) {
			if (Bool:read(src)) {
				read(to, type.members[0].type);
			} else {
				to << "<empty>";
			}
		} else if (type.flags & TypeInfo:custom) {
			// Note, we need to abort reading here. We can not continue any further since we don't
			// know what the custom type looks like.
			to << "<custom type, unable to read further>\n";
			bad = true;
		} else {
			for (m in type.members) {
				to << m.name << ": ";
				read(to, m.type);
				to << "\n";
			}
		}

		if (brackets) {
			to.dedent();
			to << type.endSym();
		}
	}

	// Find the type 'type'. Read it if necessary.
	Type findType(Nat type, StrBuf to) {
		if (t = types.at(type))
			return t;

		Byte flags = Byte:read(src);
		Str name = demangleName(Str:read(src));
		Nat parent = Nat:read(src);
		if (parent == type) {
			// if (parent == type), we should throw an exception or similar. Otherwise, we will
			// recurse infinitely when trying to read this type.
			bad = true;
			to << "<invalid parent in type description found>";
			parent = 0;
		}

		Type t(name, TypeInfo(flags.nat), parent);

		if (t.flags & TypeInfo:tuple) {
			do {
				Nat type = Nat:read(src);
			} while (type > 0) {
				t.members << Member(type, "");
			}
		} else if (t.flags & TypeInfo:maybe) {
			// Always exactly one member.
			t.members << Member(Nat:read(src), "");
		} else if (t.flags & TypeInfo:custom) {
			// Nothing to read.
		} else {
			do {
				Nat type = Nat:read(src);
			} while (type > 0) {
				Str name = Str:read(src);
				t.members << Member(type, name);
			}
		}

		types.put(type, t);

		t;
	}

	// Read a primitive type.
	Bool readPrimitive(StrBuf to, Nat type) {
		if (type == StoredId:boolId.v) {
			to << Bool:read(src);
		} else if (type == StoredId:byteId.v) {
			to << Byte:read(src) << "b";
		} else if (type == StoredId:intId.v) {
			to << Int:read(src) << "i";
		} else if (type == StoredId:natId.v) {
			to << Nat:read(src) << "n";
		} else if (type == StoredId:longId.v) {
			to << Long:read(src) << "l";
		} else if (type == StoredId:wordId.v) {
			to << Word:read(src) << "w";
		} else if (type == StoredId:floatId.v) {
			to << Float:read(src) << "f";
		} else if (type == StoredId:doubleId.v) {
			to << Double:read(src) << "d";
		} else if (type == StoredId:strId.v) {
			var quote = "\"".begin().v();
			to << "\"" << Str:read(src).escape(quote) << "\"";
		} else {
			return false;
		}

		true;
	}
}
